"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const store_1 = require("./store");
const key = 'openwhisk.history';
class HistoryModel {
    constructor() {
        this._lines = (typeof window !== 'undefined' && JSON.parse(store_1.default().getItem(key))) || [];
        this._cursor = this._lines.length;
    }
    line(idx) {
        return this._lines[idx];
    }
    slice(start, end) {
        return this._lines.slice(start, end);
    }
    get cursor() {
        return this._cursor;
    }
    guardedChange(incr) {
        const newCursor = this._cursor + incr;
        if (newCursor < 0) {
            this._cursor = 0;
        }
        else if (newCursor > this._lines.length) {
            this._cursor = this._lines.length;
        }
        else {
            this._cursor = newCursor;
        }
        return this._cursor;
    }
    wipe() {
        this._lines = [];
        store_1.default().setItem(key, JSON.stringify(this._lines));
        return true;
    }
    add(line) {
        if (this._lines.length === 0 || JSON.stringify(this._lines[this._lines.length - 1]) !== JSON.stringify(line)) {
            this._lines.push(line);
            store_1.default().setItem(key, JSON.stringify(this._lines));
        }
        this._cursor = this._lines.length;
        return this._cursor - 1;
    }
    update(cursor, updateFn) {
        updateFn(this._lines[cursor]);
        store_1.default().setItem(key, JSON.stringify(this._lines));
    }
    lineByIncr(incr) {
        return this.line(this.guardedChange(incr));
    }
    previous() {
        return this.lineByIncr(-1);
    }
    next() {
        return this.lineByIncr(+1);
    }
    first() {
        this._cursor = 0;
        return this.line(this._cursor);
    }
    last() {
        this._cursor = this._lines.length - 1;
        return this.line(this.cursor);
    }
    findIndex(filter, startIdx) {
        let filterFn;
        if (typeof filter === 'string') {
            const regexp = new RegExp(filter.replace(/([$.])/g, '\\$1'));
            filterFn = (line) => regexp.test(line.raw);
        }
        else if (filter instanceof RegExp) {
            filterFn = (line) => filter.test(line.raw);
        }
        else {
            filterFn = filter;
        }
        for (let idx = startIdx !== undefined && startIdx >= 0 ? startIdx : this._lines.length - 1; idx >= 0; idx--) {
            if (filterFn(this._lines[idx])) {
                return idx;
            }
        }
    }
    find(filter) {
        const idx = this.findIndex(filter);
        return idx !== undefined && this._lines[idx];
    }
}
exports.HistoryModel = HistoryModel;
exports.History = new HistoryModel();
exports.default = exports.History;
//# sourceMappingURL=history.js.map