"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = require("debug");
const core_1 = require("@kui-shell/core");
const limits_json_1 = require("@kui-shell/client/config.d/limits.json");
const flags_1 = require("./flags");
const fqn_1 = require("./fqn");
const options_1 = require("./options");
const command_prefix_1 = require("../command-prefix");
const fetch_file_1 = require("../../lib/util/fetch-file");
const traffic_light_1 = require("../../lib/model/traffic-light");
const states_1 = require("../../lib/model/states");
const strings = core_1.i18n('plugin-kubeui');
const debug = debug_1.default('plugin-kubeui/controller/kubectl/status');
const usage = (command) => ({
    command,
    strict: command,
    docs: 'Check the deployment status of a set of resources',
    onlyEnforceOptions: true,
    optional: [
        {
            name: '--filename',
            alias: '-f',
            file: true,
            docs: 'A kubernetes resource file or kind'
        },
        {
            name: 'resourceName',
            positional: true,
            docs: 'The name of a kubernetes resource of the given kind'
        },
        { name: '--final-state', hidden: true },
        { name: '--namespace', alias: '-n', docs: 'Inspect a specified namespace' },
        { name: '--all', alias: '-a', docs: 'Show status across all namespaces' },
        {
            name: '--multi',
            alias: '-m',
            docs: 'Display multi-cluster views as a multiple tables'
        },
        {
            name: '--response',
            docs: 'The initial response from the CRUD command'
        },
        {
            name: '--command',
            string: true,
            docs: 'The initial command from the CRUD command'
        },
        {
            name: '--watching',
            hidden: true,
            boolean: true,
            docs: 'internal use: called as part of the polling loop'
        },
        {
            name: '--watch',
            alias: '-w',
            boolean: true,
            docs: 'After listing/getting the requested object, watch for changes'
        }
    ],
    example: `kubectl ${command} @seed/cloud-functions/function/echo.yaml`
});
function getResourcesReferencedByFile(file, args) {
    return __awaiter(this, void 0, void 0, function* () {
        const [{ safeLoadAll }, raw] = yield Promise.all([Promise.resolve().then(() => require('js-yaml')), fetch_file_1.default(args.tab, file)]);
        const namespaceFromCommandLine = options_1.getNamespace(args) || 'default';
        const models = safeLoadAll(raw);
        return models.map(({ apiVersion, kind, metadata: { name, namespace = namespaceFromCommandLine } }) => {
            const { group, version } = fqn_1.versionOf(apiVersion);
            return {
                group,
                version,
                kind,
                name,
                namespace
            };
        });
    });
}
function getResourcesReferencedByCommandLine(argvRest, args) {
    const [kind, nameGroupVersion] = argvRest;
    const [name, group, version] = nameGroupVersion.split(/\./);
    const namespace = options_1.getNamespace(args) || 'default';
    return [{ group, version, kind, name, namespace }];
}
function isResourceReady(row, finalState) {
    const status = row.attributes.find(_ => _.key === 'STATUS');
    if (status !== undefined) {
        return states_1.isDone(status.value, finalState);
    }
    else {
        const ready = row.attributes.find(_ => _.key === 'READY');
        if (ready !== undefined) {
            const [nReady, nTotal] = ready.value.split(/\//);
            return nReady && nTotal && nReady === nTotal;
        }
    }
    return true;
}
function clone(row) {
    const copy = Object.assign({}, row);
    copy.attributes = row.attributes.map(_ => Object.assign({}, _));
    return copy;
}
class StatusPoller {
    constructor(tab, ref, row, finalState, done, pusher, contextArgs, command, pollInterval = 1000, ladder = StatusPoller.calculateLadder(pollInterval)) {
        this.tab = tab;
        this.ref = ref;
        this.row = row;
        this.finalState = finalState;
        this.done = done;
        this.pusher = pusher;
        this.contextArgs = contextArgs;
        this.command = command;
        this.ladder = ladder;
        this.pollOnce(0);
    }
    pollOnce(iter) {
        return __awaiter(this, void 0, void 0, function* () {
            const sleepTime = iter < this.ladder.length ? this.ladder[iter] : this.ladder[this.ladder.length - 1];
            debug('pollOnce', this.ref, sleepTime, fqn_1.fqnOfRef(this.ref));
            try {
                const table = yield this.tab.REPL.qexec(`${this.command} get ${fqn_1.fqnOfRef(this.ref)} ${this.contextArgs}`);
                debug('pollOnce table', table);
                if (table && table.body && table.body.length === 1) {
                    const row = table.body[0];
                    const isReady = isResourceReady(row, this.finalState);
                    const newStatusAttr = row.attributes.find(_ => _.key === 'STATUS') || row.attributes.find(_ => _.key === 'READY');
                    const rowForUpdate = clone(this.row);
                    const statusAttr = rowForUpdate.attributes.find(({ key }) => key === 'STATUS');
                    statusAttr.value = newStatusAttr ? newStatusAttr.value : 'Ready';
                    if (isReady) {
                        statusAttr.css = traffic_light_1.default.Green;
                    }
                    this.pusher.update(rowForUpdate);
                    if (isReady) {
                        debug('resource is ready', this.ref, this.row, newStatusAttr);
                        this.done();
                        return;
                    }
                }
                else {
                    console.error('unexpected tabular response in poller', table);
                }
            }
            catch (error) {
                const err = error;
                if (err.code === 404 && this.finalState === states_1.FinalState.OfflineLike) {
                    this.pusher.offline(this.ref.name);
                    this.done();
                    return;
                }
            }
            this.timer = setTimeout(() => this.pollOnce(iter + 1), sleepTime);
        });
    }
    static calculateLadder(initial) {
        const finalPolling = limits_json_1.tablePollingInterval || 5000;
        const ladder = [initial];
        let current = initial;
        while (current < finalPolling) {
            if (current < 1000) {
                current = current + 250 < 1000 ? current + 250 : 1000;
                ladder.push(current);
            }
            else {
                ladder.push(current);
                current = current + 2000 < finalPolling ? current + 2000 : finalPolling;
                ladder.push(current);
            }
        }
        return ladder;
    }
    abort() {
        if (this.timer) {
            clearTimeout(this.timer);
        }
    }
}
class StatusWatcher {
    constructor(tab, resourcesToWaitFor, finalState, contextArgs, command) {
        this.tab = tab;
        this.resourcesToWaitFor = resourcesToWaitFor;
        this.finalState = finalState;
        this.contextArgs = contextArgs;
        this.command = command;
        this.pollers = [];
    }
    abort() {
        this.pollers.forEach(poller => {
            if (poller) {
                poller.abort();
            }
        });
    }
    init(pusher) {
        let countdown = this.resourcesToWaitFor.length;
        const done = () => {
            if (--countdown === 0) {
                debug('all resources are ready');
                pusher.done();
                for (let idx = 0; idx < this.pollers.length; idx++) {
                    this.pollers[idx] = undefined;
                }
            }
        };
        this.resourcesToWaitFor
            .map((_, idx) => {
            const row = this.initialBody[idx];
            return new StatusPoller(this.tab, _, row, this.finalState, done, pusher, this.contextArgs, this.command);
        })
            .forEach(_ => {
            this.pollers.push(_);
        });
    }
    nsAttr(ns, anyNonDefaultNamespaces) {
        return !anyNonDefaultNamespaces ? [] : [{ key: 'NAMESPACE', value: ns }];
    }
    initialTable() {
        const anyNonDefaultNamespaces = this.resourcesToWaitFor.some(({ namespace }) => namespace !== 'default');
        this.initialBody = this.resourcesToWaitFor.map(ref => {
            const { group = '', version = '', kind, name, namespace } = ref;
            return {
                name,
                onclick: `${this.command}  get ${fqn_1.fqnOfRef(ref)} -o yaml`,
                onclickSilence: true,
                attributes: this.nsAttr(namespace, anyNonDefaultNamespaces).concat([
                    {
                        key: 'KIND',
                        value: kind + (group.length > 0 ? `.${version}.${group}` : ''),
                        outerCSS: '',
                        css: ''
                    },
                    { key: 'STATUS', tag: 'badge', value: strings('Pending'), outerCSS: '', css: traffic_light_1.default.Yellow }
                ])
            };
        });
        const initialHeader = {
            name: 'NAME',
            attributes: this.initialBody[0].attributes.map(({ key, outerCSS }) => ({ value: key, outerCSS }))
        };
        return {
            header: initialHeader,
            body: this.initialBody,
            watch: this
        };
    }
}
const doStatus = (command) => (args) => __awaiter(void 0, void 0, void 0, function* () {
    const rest = args.argvNoOptions.slice(args.argvNoOptions.indexOf('status') + 1);
    const commandArg = command || args.parsedOptions.command;
    const file = options_1.fileOf(args);
    const contextArgs = options_1.getContextForArgv(args);
    try {
        const resourcesToWaitFor = file
            ? yield getResourcesReferencedByFile(file, args)
            : getResourcesReferencedByCommandLine(rest, args);
        debug('resourcesToWaitFor', resourcesToWaitFor);
        const finalState = args.parsedOptions['final-state'];
        return new StatusWatcher(args.tab, resourcesToWaitFor, finalState, contextArgs, commandArg).initialTable();
    }
    catch (err) {
        console.error('error constructing StatusWatcher', err);
        const initialResponse = args.parsedOptions.response;
        return initialResponse;
    }
});
exports.default = (registrar) => {
    const opts = Object.assign({
        usage: usage('status')
    }, flags_1.flags(['watching']));
    registrar.listen(`/${command_prefix_1.default}/kubectl/status`, doStatus('kubectl'), opts);
    registrar.listen(`/${command_prefix_1.default}/k/status`, doStatus('k'), opts);
    registrar.listen(`/${command_prefix_1.default}/status`, doStatus(''), opts);
};
//# sourceMappingURL=status.js.map