import { ParsedOptions } from '../command';
import { Tab } from '../../webapp/tab';
import { Table } from '../../webapp/models/table';
import { Entity, MetadataBearing } from '../entity';
import { CustomSpec } from '../../webapp/views/sidecar-core';
import { SidecarMode } from '../../webapp/bottom-stripe';
/**
 * A `ScalarResource` is Any kind of resource that is directly
 * represented, as opposed to being implicitly represented by a
 * function call.
 *
 */
export declare type ScalarResource = CustomSpec | string | HTMLElement | Table;
export interface ScalarContent<T = ScalarResource> {
    content: T;
}
export declare function isScalarContent<T extends MetadataBearing>(entity: Entity | ScalarResource | ScalarContent | Content<T> | MetadataBearing | SidecarMode): entity is ScalarContent;
/**
 * Supported String content types
 *
 */
declare type SupportedStringContent = 'yaml' | 'text/markdown' | 'text/html' | 'json';
/**
 * Trait that provides optional contentType for string content
 *
 */
interface WithOptionalContentType<ContentType = SupportedStringContent> {
    contentType?: ContentType;
}
/**
 * Special case of `ScalarContent` for strings; string content may
 * optionally provide a `contentType`.
 *
 */
export declare type StringContent<ContentType = SupportedStringContent> = ScalarContent<string> & WithOptionalContentType<ContentType>;
export declare function isStringWithOptionalContentType<T extends MetadataBearing>(entity: Entity | Content<T> | MetadataBearing | SidecarMode): entity is StringContent;
/**
 * `Content` as `FunctionThatProducesContent<T>` is a function that
 * takes a `T` and produces either a resource or some { content,
 * contentType } wrapper.
 *
 */
export declare type FunctionThatProducesContent<T extends MetadataBearing> = (tab: Tab, entity: T, args: {
    argvNoOptions: string[];
    parsedOptions: ParsedOptions;
}) => ScalarResource | ScalarContent | CommandStringContent | Promise<ScalarResource> | Promise<ScalarContent>;
export interface FunctionContent<T extends MetadataBearing> {
    content: FunctionThatProducesContent<T>;
}
export declare function isFunctionContent<T extends MetadataBearing>(content: Entity | Content<T> | MetadataBearing | SidecarMode): content is FunctionContent<T>;
/**
 * `Content` as `CommandStringContent` will be executed via a
 * REPL.qexec; the command is assumed to return either
 * `ScalarRersource` or `ScalarContent`.
 *
 */
declare type CommandStringContent = WithOptionalContentType<SupportedStringContent> & {
    contentFrom: string;
};
export declare function isCommandStringContent<T extends MetadataBearing>(content: ScalarResource | Content<T> | MetadataBearing | SidecarMode): content is CommandStringContent;
/**
 * The classes of supported `Content` are:
 * - `ScalarContent`: such as pre-formatted HTML, or or `Table` model
 * - `StringContent`: a string with an optional `contentType`
 * - `FunctionContent<T>`: a function that takes a `T` and produces `ScalarContent`
 * - `CommandStringContent`: a string to be executed as a Kui command, and producing `ScalarContent`
 *
 */
export declare type Content<T extends MetadataBearing> = ScalarContent | StringContent | FunctionContent<T> | CommandStringContent;
export declare function hasContent<T extends MetadataBearing>(resource: ScalarResource | Content<T> | SidecarMode): resource is Content<T>;
export {};
