/*
 * Copyright 2017-19 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import Debug from 'debug';
const debug = Debug('webapp/oops');
debug('loading');
import { getPrompt } from './prompt';
import { setStatus } from './status';
import { popupListen } from './listen';
import { installBlock } from './block';
import { isPopup } from './popup-core';
import { renderPopupContent, createPopupContentContainer } from './popup';
import { isHTML } from '../util/types';
import { oopsMessage } from '../core/oops';
import UsageError from '../core/usage-error';
import { isHideError, isCodedError } from '../models/errors';
function isPopupError(err) {
    return err.content !== undefined;
}
/**
 * Handle command execution errors
 *
 */
export const oops = (command, block, nextBlock) => (err) => __awaiter(void 0, void 0, void 0, function* () {
    if (!block)
        return; // we're not attached to a prompt right now
    if (!nextBlock) {
        nextBlock = block.cloneNode(true);
        nextBlock.querySelector('input').value = '';
    }
    if (getPrompt(block).value === '') {
        // e.g. we want qexec with reportErrors:true show command in repl
        getPrompt(block).value = command;
    }
    setStatus(block, "error" /* error */);
    const resultDom = isPopup() ? createPopupContentContainer(['error']) : block.querySelector('.repl-result');
    const oopsDom = document.createElement('div');
    oopsDom.className = 'oops';
    resultDom.appendChild(oopsDom);
    if (isHideError(err)) {
        // we were instructed not to show any message
        debug('we were instructed to hide this error', err);
        oopsDom.classList.add('hidden');
    }
    else if (UsageError.isUsageError(err)) {
        oopsDom.appendChild(yield UsageError.getFormattedMessage(err));
        /* } else if (isHTML(err.message)) {
        // err.message is a DOM
        oopsDom.appendChild(err.message) */
        /* } else if (err.html) {
        // pre-rendered HTML
        oopsDom.classList.add('oops-as-html')
        oopsDom.appendChild(err.html) */
        /* } else if (err.message && err.message.then) {
        err.message.then(message => {
          err.message = message
          oops(command, block, nextBlock)(err)
        })
        return */
    }
    else if (isHTML(err)) {
        // err is a DOM
        oopsDom.appendChild(err);
    }
    else {
        // we'll go with our formatted message
        // wrap in a span so that drag text selection works; see shell issue #249
        const message = oopsMessage(err);
        const span = document.createElement('pre');
        span.appendChild(document.createTextNode(message));
        oopsDom.appendChild(span);
    }
    // add the http status code, if we have it (helps with testing)
    if (isCodedError(err)) {
        oopsDom.setAttribute('data-status-code', (err.statusCode || err.code).toString());
    }
    else {
        oopsDom.setAttribute('data-status-code', '0');
    }
    if (resultDom.hasAttribute('data-stream')) {
        // then the command has been streaming its output; copy any such output
        // over to the oops dom
        const streamingOutput = resultDom.querySelector('.streaming-output');
        if (streamingOutput) {
            oopsDom.appendChild(streamingOutput);
        }
    }
    if (isPopup()) {
        yield renderPopupContent(command, (isPopupError(err) && err.content) || resultDom, {}, isPopupError(err) && err.modes);
        popupListen(undefined, command);
    }
    installBlock(block.parentNode, block, nextBlock)();
    // indicate that we've already rendered the block
    return false;
});
export const showHelp = (command, block, nextBlock, error) => {
    // if the message says command not found, then add on the "enter help to see your options" as a suffix
    const baseMessage = 'Enter help to see your options.';
    if (error.message && error.message === 'Command not found')
        error.message += `\n${baseMessage}`;
    return oops(command, block, nextBlock)(error) && false;
};
//# sourceMappingURL=oops.js.map