"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const debug_1 = require("debug");
const child_process_1 = require("child_process");
const core_1 = require("@kui-shell/core");
const exec_1 = require("../util/exec");
const json_1 = require("../util/json");
const usage_helpers_1 = require("../util/usage-helpers");
const catchall_1 = require("./catchall");
const strings = core_1.i18n('plugin-bash-like');
const debug = debug_1.default('plugins/bash-like/cmds/general');
exports.doExec = (cmdLine, execOptions) => new Promise((resolve, reject) => __awaiter(void 0, void 0, void 0, function* () {
    const options = {
        maxBuffer: 1 * 1024 * 1024,
        env: Object.assign({}, process.env, execOptions['env'] || {})
    };
    if (process.env.SHELL) {
        options.shell = process.env.SHELL;
    }
    const proc = child_process_1.exec(cmdLine, options);
    let rawOut = '';
    let rawErr = '';
    proc.stdout.on('data', (data) => __awaiter(void 0, void 0, void 0, function* () {
        const out = data.toString();
        if (execOptions.stdout) {
            execOptions.stdout(data);
        }
        else {
            rawOut += out;
        }
    }));
    proc.stderr.on('data', data => {
        rawErr += data;
        if (execOptions.stderr) {
            execOptions.stderr(data.toString());
        }
    });
    proc.on('close', (exitCode) => __awaiter(void 0, void 0, void 0, function* () {
        if (exitCode === 0) {
            if (execOptions && execOptions['json']) {
                try {
                    resolve(JSON.parse(rawOut));
                }
                catch (err) {
                    const error = new Error('unexpected non-JSON');
                    error['value'] = rawOut;
                    reject(error);
                }
            }
            else if (execOptions && execOptions.raw) {
                resolve(rawOut);
            }
            else if (!rawOut && !rawErr) {
                resolve(true);
            }
            else {
                const json = json_1.extractJSON(rawOut);
                if (json) {
                    json['type'] = 'shell';
                    json['verb'] = 'get';
                    resolve(json);
                }
                else {
                    resolve(rawOut);
                }
            }
        }
        else {
            debug('non-zero exit code', exitCode);
            const cleanErr = rawErr.replace(/(^\/[^/]+\/[^:]+: )(line \d+: )?/, '');
            try {
                exec_1.handleNonZeroExitCode(cmdLine, exitCode, rawOut, cleanErr, execOptions);
            }
            catch (err) {
                reject(err);
            }
        }
    }));
}));
const doShell = (argv, args) => new Promise((resolve, reject) => __awaiter(void 0, void 0, void 0, function* () {
    const { execOptions, REPL } = args;
    if (core_1.inBrowser()) {
        reject(new Error('Local file access not supported when running in a browser'));
    }
    const shell = yield Promise.resolve().then(() => require('shelljs'));
    if (argv.length < 2) {
        reject(new Error('Please provide a bash command'));
    }
    const cmd = argv[1];
    debug('argv', argv);
    debug('cmd', cmd);
    if (shell[cmd] && (core_1.inBrowser() || (cmd !== 'mkdir' && cmd !== 'echo'))) {
        const args = argv.slice(2);
        if (process.env.OLDPWD === undefined) {
            process.env.OLDPWD = '';
        }
        const OLDPWD = shell.pwd();
        if (cmd === 'cd' && args[0] === '-') {
            args[0] = process.env.OLDPWD;
        }
        if (!args.find(arg => arg.charAt(0) === '-') &&
            !args.find(arg => arg === '>') &&
            cmd !== 'ls') {
            debug('using internal shelljs', cmd, args);
            const output = shell[cmd](args);
            if (cmd === 'cd') {
                process.env.OLDPWD = OLDPWD;
                process.env.PWD = output;
                if (output.code === 0) {
                    resolve(shell.pwd().toString());
                }
                else {
                    reject(new Error(output.stderr));
                }
            }
            else {
                if (output && output.length > 0) {
                    if (execOptions && execOptions['json']) {
                        resolve(JSON.parse(output));
                    }
                    else {
                        resolve(output.toString());
                    }
                }
                else {
                    resolve(true);
                }
            }
        }
    }
    const rest = argv.slice(1);
    const cmdLine = rest.map(_ => REPL.encodeComponent(_)).join(' ');
    debug('cmdline', cmdLine, rest);
    exports.doExec(cmdLine, execOptions).then(resolve, reject);
}));
const usage = {
    cd: {
        strict: 'cd',
        command: 'cd',
        title: strings('cdUsageTitle'),
        header: strings('cdUsageHeader'),
        optional: usage_helpers_1.localFilepath
    }
};
const cd = (args) => {
    const dir = args.REPL.split(args.command, true, true)[1] || '';
    debug('cd dir', dir);
    return doShell(['!', 'cd', dir], args).catch(err => {
        err['code'] = 500;
        throw err;
    });
};
const bcd = ({ command, execOptions, REPL }) => __awaiter(void 0, void 0, void 0, function* () {
    const pwd = yield REPL.qexec(command.replace(/^cd/, 'kuicd'), undefined, undefined, execOptions);
    debug('pwd', pwd);
    process.env.PWD = pwd;
    return pwd;
});
const specialHandler = (args) => {
    if (args.execOptions.type === core_1.ExecType.TopLevel) {
        throw new Error('this command is intended for internal consumption only');
    }
    return catchall_1.dispatchToShell(args);
};
exports.default = (commandTree) => {
    commandTree.listen('/!', catchall_1.dispatchToShell, {
        docs: 'Execute a UNIX shell command',
        noAuthOk: true,
        requiresLocal: true
    });
    commandTree.listen('/sendtopty', specialHandler, {
        docs: 'Execute a UNIX shell command with a PTY',
        noAuthOk: true,
        inBrowserOk: true,
        hidden: true
    });
    commandTree.listen('/kuicd', cd, {
        noAuthOk: true
    });
    if (!core_1.inBrowser()) {
        commandTree.listen('/cd', cd, {
            usage: usage.cd,
            noAuthOk: true
        });
    }
    else {
        commandTree.listen('/cd', bcd, {
            usage: usage.cd,
            noAuthOk: true,
            inBrowserOk: true
        });
    }
};
//# sourceMappingURL=bash-like.js.map