/*
 * Copyright 2019 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { Common, CLI, ReplExpect } from '@kui-shell/test';
import { readFile } from 'fs-extra';
import { execSync } from 'child_process';
/** skip the tests if we aren't doing a webpack+proxy test run */
const runTheTests = process.env.MOCHA_RUN_TARGET === 'webpack' && process.env.KUI_USE_PROXY === 'true';
const pit = runTheTests ? it : xit;
/** the proxy should come back online within 10 seconds; add some slop */
const timeItTakesForProxyToComeBack = 30000;
describe('pty session status offline after start', function () {
    before(Common.before(this));
    after(Common.after(this));
    pit('should echo hi', () => CLI.command('echo hi', this.app)
        .then(ReplExpect.okWithString('hi'))
        .catch(Common.oops(this)));
    pit('should not be showing an offline indicator', () => {
        return this.app.client
            .waitForExist('.kui--plugin-bash-like--pty-offline-indicator', 5000, true)
            .catch(Common.oops(this));
    });
    pit('should kill the proxy and show an offline indicator', () => __awaiter(this, void 0, void 0, function* () {
        try {
            const proxyPid = (yield readFile('/tmp/kuiproxy.pid')).toString();
            execSync(`kill -HUP ${proxyPid}`);
            yield this.app.client.waitForExist('.kui--plugin-bash-like--pty-offline-indicator');
            yield this.app.client.waitForExist('.kui--plugin-bash-like--pty-offline-indicator', timeItTakesForProxyToComeBack, true);
        }
        catch (err) {
            Common.oops(this)(err);
        }
    }));
});
describe('pty session status offline at start', function () {
    before(Common.before(this, {
        noProxySessionWait: true,
        beforeStart: () => __awaiter(this, void 0, void 0, function* () {
            if (runTheTests) {
                console.log('killing the proxy');
                const proxyPid = (yield readFile('/tmp/kuiproxy.pid')).toString();
                execSync(`kill -HUP ${proxyPid}`);
                yield new Promise(resolve => setTimeout(resolve, 2000));
            }
        })
    }));
    after(Common.after(this));
    // we expect the offline indicator initially
    pit('should be showing an offline indicator at startup because we killed the proxy', () => {
        return this.app.client.waitForExist('.kui--plugin-bash-like--pty-offline-indicator').catch(Common.oops(this, true));
    });
    // but we also expect it to go away eventually
    pit('should eventually not show an offline indicator', () => __awaiter(this, void 0, void 0, function* () {
        try {
            yield this.app.client.waitForExist('.kui--plugin-bash-like--pty-offline-indicator', 20000, true);
            yield CLI.waitForRepl(this.app);
        }
        catch (err) {
            yield Common.oops(this, true)(err);
        }
    }));
    pit('should be not showing an offline indicator at startup because the proxy auto-restarts', () => {
        // then it should come back online within 10 seconds; add a few seconds of slop
        return this.app.client
            .waitForExist('.kui--plugin-bash-like--pty-offline-indicator', timeItTakesForProxyToComeBack, true)
            .catch(Common.oops(this, true));
    });
    pit('should cd to the test dir', () => CLI.command(`cd ${process.env.TEST_ROOT}`, this.app)
        .then(ReplExpect.okWithString('packages/test'))
        .catch(Common.oops(this, true)));
    pit('should ls now that the proxy is online', () => CLI.command('ls -l ../..', this.app)
        .then(ReplExpect.okWith('package.json'))
        .catch(Common.oops(this, true)));
    pit('should echo hi now that the proxy is online', () => CLI.command('echo hi', this.app)
        .then(ReplExpect.okWithString('hi'))
        .catch(Common.oops(this, true)));
});
//# sourceMappingURL=pty-session-status.js.map