/*
 * Copyright 2018 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
/* eslint-disable @typescript-eslint/explicit-member-accessibility */
import Debug from 'debug';
import { inBrowser, eventBus, inBottomInputMode, getCurrentPrompt, getBlockOfPrompt, getCurrentPromptLeft, setUsingCustomPrompt, unsetUsingCustomPrompt, getTabFromTarget, KeyCodes, isCursorMovement, History } from '@kui-shell/core';
const debug = Debug('core-support/history/reverse-i-search');
// TODO externalize
const strings = {
    prompt: "(reverse-i-search$1)`$2': "
};
/** state of the reverse-i-search */
class ActiveISearch {
    constructor(tab) {
        this.isSearchActive = true;
        this.currentSearchIdx = -1;
        this.tab = tab;
        this.prompt = getCurrentPrompt(tab);
        this.promptLeft = getCurrentPromptLeft(tab);
        this.placeholder = document.createElement('span');
        this.placeholderFixedPart = document.createElement('span');
        this.placeholderFixedPart.innerText = strings.prompt.replace(/\$1/, '').replace(/\$2/, this.prompt.value);
        this.placeholder.appendChild(this.placeholderFixedPart);
        this.placeholder.classList.add('repl-temporary');
        this.placeholder.classList.add('normal-text');
        this.placeholder.classList.add('monospace');
        this.placeholderFixedPart.classList.add('smaller-text');
        this.placeholderFixedPart.classList.add('small-right-pad');
        this.promptLeft.appendChild(this.placeholder);
        setUsingCustomPrompt(getBlockOfPrompt(this.prompt));
        //    this.prompt.style.opacity = '0'
        //    this.prompt.style.width = '0'
        this.placeholderContentPart = document.createElement('span'); // container for Typed and Matched
        this.placeholderTypedPart = document.createElement('strong'); // what the user has typed; e.g. "is" in "history"
        this.placeholderMatchedPrefixPart = document.createElement('span'); // what was matched, but not typed; e.g. "h" in "history"
        this.placeholderMatchedSuffixPart = document.createElement('span'); // what was matched, but not typed; e.g. "tory" in "history"
        this.placeholderContentPart.appendChild(this.placeholderMatchedPrefixPart);
        this.placeholderContentPart.appendChild(this.placeholderTypedPart);
        this.placeholderContentPart.appendChild(this.placeholderMatchedSuffixPart);
        this.placeholderContentPart.classList.add('repl-input-like');
        this.placeholder.appendChild(this.placeholderContentPart);
        this.placeholderTypedPart.classList.add('red-text');
        this.placeholderTypedPart.classList.add('kui--prompt-like');
        this.placeholderMatchedPrefixPart.classList.add('slightly-deemphasize');
        this.placeholderMatchedSuffixPart.classList.add('slightly-deemphasize');
        this.currentOnInput = this.prompt.oninput;
        this.prompt.oninput = this.doSearch.bind(this);
        this.currentOnKeypress = this.prompt.onkeypress;
        this.prompt.onkeypress = this.maybeComplete.bind(this);
    }
    /**
     * For various reasons, user has cancelled a reverse-i-search.
     *
     */
    cancelISearch(evt) {
        const isCtrlC = evt && evt.keyCode === KeyCodes.C && evt.ctrlKey;
        this.tab['_kui_active_i_search'] = undefined;
        if (this.isSearchActive) {
            this.isSearchActive = false;
            if (!isCtrlC || inBottomInputMode) {
                unsetUsingCustomPrompt(getBlockOfPrompt(this.prompt));
                if (this.placeholder.parentNode) {
                    this.placeholder.parentNode.removeChild(this.placeholder);
                }
                this.prompt.onkeypress = this.currentOnKeypress;
                this.prompt.oninput = this.currentOnInput;
                this.prompt.focus();
            }
        }
    }
    /**
     * Attempt to initiate or extend a search
     *
     */
    doSearch(evt) {
        return __awaiter(this, void 0, void 0, function* () {
            debug('doSearch', evt);
            if (evt.inputType === 'deleteContentBackward') {
                // if the user hits Backspace, reset currentSearchIdx
                // TODO confirm that this is the behavior of bash
                this.currentSearchIdx = -1;
                this.placeholderFixedPart.innerText = strings.prompt.replace(/\$1/, ``).replace(/\$2/, this.prompt.value);
            }
            // where do we want to start the search? if the user is just
            // typing, then start from the end of history; if the user hit
            // ctrl+r, then they want to search for the next match
            const userHitCtrlR = evt.ctrlKey && evt.code === 'KeyR';
            const startIdx = userHitCtrlR ? this.currentSearchIdx - 1 : -1;
            const history = yield History(this.tab);
            const newSearchIdx = this.prompt.value && history.findIndex(this.prompt.value, startIdx);
            debug('search index', this.prompt.value, newSearchIdx);
            if (newSearchIdx > 0) {
                this.currentSearchIdx = newSearchIdx;
                this.placeholderFixedPart.innerText = strings.prompt
                    .replace(/\$1/, '') // ` ${newSearchIdx}`
                    .replace(/\$2/, this.prompt.value);
                const newValue = history.line(this.currentSearchIdx).raw;
                debug('newValue', newValue);
                const caretPosition = newValue.indexOf(this.prompt.value) + 1;
                debug('caretPosition', caretPosition);
                const matchedPrefix = newValue.substring(0, caretPosition - 1);
                const matchedSuffix = newValue.substring(caretPosition + this.prompt.value.length - 1);
                debug('matchedPrefix', matchedPrefix, newValue, caretPosition);
                debug('matchedSuffix', matchedSuffix);
                this.placeholderTypedPart.innerText = this.prompt.value.replace(/ /g, '_'); // show matched whitespaces with an underscore
                this.placeholderMatchedPrefixPart.innerText = matchedPrefix;
                this.placeholderMatchedSuffixPart.innerText = matchedSuffix;
                this.placeholderContentPart.setAttribute('data-full-match', newValue);
            }
            else if (!userHitCtrlR) {
                // if we found no match, reset the match text, unless the user
                // is using repeated ctrl+R to search backwards; in this case,
                // let's continue to display the previous match if no new match
                // is found
                this.placeholderTypedPart.innerText = '';
                this.placeholderMatchedPrefixPart.innerText = '';
                this.placeholderMatchedSuffixPart.innerText = '';
                this.placeholderFixedPart.innerText = strings.prompt.replace(/\$1/, ``).replace(/\$2/, this.prompt.value);
            }
            else {
                this.placeholderFixedPart.classList.add('alert-pulse');
                setTimeout(() => this.placeholderFixedPart.classList.remove('alert-pulse'), 1000);
            }
        });
    }
    /** fill in the result of a search */
    completeSearch() {
        debug('completing search');
        this.prompt.value = this.placeholderContentPart.getAttribute('data-full-match');
        this.cancelISearch();
    }
    /**
     * User hits enter while in i-search mode
     *
     */
    maybeComplete(evt) {
        if (this.isSearchActive) {
            if (evt.keyCode === KeyCodes.ENTER) {
                this.completeSearch();
                this.prompt.dispatchEvent(new KeyboardEvent(evt.type, evt));
            }
        }
    }
}
/**
 * Listen for ctrl+R
 *
 */
function registerListener() {
    if (typeof document === 'undefined') {
        // fail-safe, in case we have no DOM
        return;
    }
    if (inBottomInputMode) {
        eventBus.on('/core/cli/install-block', (tab) => {
            const activeSearch = tab['_kui_active_i_search'];
            if (activeSearch) {
                activeSearch.cancelISearch();
            }
        });
    }
    /**
     * Listen for ctrl+r
     *
     */
    document.getElementsByTagName('body')[0].addEventListener('keyup', (evt) => {
        //
        // we want ctrl+R; but if we're in a browser and on linux or
        // windows, then ctrl+R will result in a browser reload :(
        //
        // Note: even if not in a browser (i.e. running in electron mode),
        // on linux and windows we have to be careful not to use the
        // default reload keyboard shortcut; see app/src/main/menu.js
        //
        // re: RUNNING_SHELL_TEST; there seems to be some weird bug here; on linux, the ctrlKey modifier becomes sticky
        if (!document.activeElement.classList.contains('repl-input-element')) {
            // if we aren't focused on a repl input, don't bother
        }
        else if (evt.ctrlKey &&
            (process.platform === 'darwin' ||
                /Macintosh/.test(navigator.userAgent) ||
                (!inBrowser() && !process.env.RUNNING_SHELL_TEST) ||
                evt.metaKey)) {
            const tab = getTabFromTarget(evt.srcElement);
            const activeSearch = tab['_kui_active_i_search'];
            if (evt.keyCode === KeyCodes.R) {
                debug('got ctrl+r');
                if (activeSearch) {
                    debug('continuation of existing reverse-i-search');
                    activeSearch.doSearch(evt);
                }
                else {
                    debug('new reverse-i-search');
                    tab['_kui_active_i_search'] = new ActiveISearch(tab);
                }
            }
            else if (activeSearch && isCursorMovement(evt)) {
                activeSearch.completeSearch();
            }
            else if (activeSearch) {
                // with ctrl key down, let any other keycode result in cancelling the outstanding i-search
                debug('cancel', evt.keyCode);
                activeSearch.cancelISearch(evt);
            }
        }
    });
}
/**
 * Listen for ctrl+R, with a fail-safe try/catch
 *
 */
export default () => {
    try {
        registerListener();
    }
    catch (err) {
        // console.error('Not running in electron environment')
        debug('Not running in electron environment');
    }
};
//# sourceMappingURL=reverse-i-search.js.map