var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import Debug from 'debug';
import { encodeComponent, i18n } from '@kui-shell/core';
import TrafficLight from '../../model/traffic-light';
import { getCommandFromArgs } from '../../util/util';
const strings = i18n('plugin-kubeui');
const debug = Debug('k8s/view/modes/containers');
const headerModel = (pod) => {
    const statuses = pod.status && pod.status.containerStatuses;
    const specAttrs = [{ value: 'PORTS', outerCSS: 'header-cell pretty-narrow' }];
    const statusAttrs = !statuses
        ? []
        : [
            { value: 'RESTARTS', outerCSS: 'header-cell very-narrow' },
            { value: 'READY', outerCSS: 'header-cell very-narrow' },
            { value: 'STATE', outerCSS: 'header-cell pretty-narrow' },
            { value: 'MESSAGE', outerCSS: 'header-cell' }
        ];
    return {
        type: 'container',
        name: 'IMAGE',
        outerCSS: 'header-cell not-too-wide',
        attributes: specAttrs.concat(statusAttrs)
    };
};
const showLogs = (tab, { pod, container }, args) => {
    const podName = encodeComponent(pod.metadata.name);
    const containerName = encodeComponent(container.name);
    const ns = encodeComponent(pod.metadata.namespace);
    return `${getCommandFromArgs(args)} logs ${podName} ${containerName} -n ${ns}`;
};
const bodyModel = (tab, pod, args) => {
    const statuses = pod.status && pod.status.containerStatuses;
    const bodyModel = pod.spec.containers
        .map(container => {
        const status = statuses && statuses.find(_ => _.name === container.name);
        if (!status) {
            return;
        }
        debug('container status', container.name, status.restartCount, status);
        const stateKey = Object.keys(status.state)[0];
        const stateBody = status.state[stateKey];
        const statusAttrs = !status
            ? []
            : [
                {
                    key: 'restartCount',
                    value: status.restartCount,
                    outerCSS: 'very-narrow'
                },
                {
                    key: 'ready',
                    value: status.ready,
                    fontawesome: status.ready ? 'fas fa-check-circle' : 'far fa-dot-circle',
                    css: status.ready ? 'green-text' : 'yellow-text'
                },
                {
                    key: 'state',
                    value: stateKey,
                    tag: 'badge',
                    outerCSS: 'capitalize',
                    css: stateKey === 'running'
                        ? TrafficLight.Green
                        : stateKey === 'terminated'
                            ? TrafficLight.Red
                            : TrafficLight.Yellow
                },
                {
                    key: 'message',
                    outerCSS: 'smaller-text not-too-wide',
                    value: stateBody.startedAt || stateBody.reason
                }
            ];
        const portsAttr = {
            key: 'ports',
            outerCSS: 'not-too-wide',
            value: (container.ports || []).map(({ containerPort, protocol }) => `${containerPort}/${protocol}`).join(' ')
        };
        const specAttrs = [portsAttr];
        return {
            type: 'container',
            name: container.name,
            onclick: showLogs(tab, { pod, container }, args),
            attributes: specAttrs.concat(statusAttrs)
        };
    })
        .filter(_ => _);
    debug('body model', bodyModel);
    return bodyModel;
};
function renderContainers(tab, resource, args) {
    return __awaiter(this, void 0, void 0, function* () {
        const fetchPod = `${getCommandFromArgs(args)} get pod ${encodeComponent(resource.metadata.name)} -n "${resource.metadata.namespace}" -o json`;
        debug('issuing command', fetchPod);
        try {
            const podResource = yield tab.REPL.qexec(fetchPod, undefined, undefined, { raw: true });
            debug('renderContainers.response', podResource);
            return {
                header: headerModel(podResource),
                body: bodyModel(tab, podResource, args),
                noSort: true,
                title: 'Containers'
            };
        }
        catch (err) {
            if (err.code === 404) {
                return { body: [] };
            }
            else {
                throw err;
            }
        }
    });
}
function hasContainers(resource) {
    return resource.spec && resource.spec.containers;
}
export const containersMode = {
    when: hasContainers,
    mode: {
        mode: 'containers',
        label: strings('containers'),
        content: renderContainers
    }
};
export default containersMode;
//# sourceMappingURL=containers.js.map