var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { execSync } from 'child_process';
import { readFileSync, writeFileSync } from 'fs';
import { safeDump, safeLoad as parseYAML } from 'js-yaml';
import * as path from 'path';
import * as assert from 'assert';
import { expandHomeDir } from '@kui-shell/core';
import { Common, CLI, ReplExpect, SidecarExpect, Selectors } from '@kui-shell/test';
import { waitForGreen, waitForRed, createNS, waitTillNone } from '@kui-shell/plugin-kubeui/tests/lib/k8s/utils';
const synonyms = ['kubectl'];
const initialContext = execSync('kubectl config current-context')
    .toString()
    .trim();
Common.localDescribe('electron context switching', function () {
    before(Common.before(this));
    after(Common.after(this, () => {
        console.log(`switching back to initial context ${initialContext}`);
        execSync(`kubectl config use-context ${initialContext}`);
        console.log(`switched to ${execSync('kubectl config current-context')}`);
    }));
    synonyms.forEach(kubectl => {
        const deleteIt = (name, errOk = false) => {
            it(`should delete the namespace ${name} via ${kubectl}`, () => {
                return CLI.command(`${kubectl} delete namespace ${name}`, this.app)
                    .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME(name), errOk }))
                    .then(selector => waitForRed(this.app, selector))
                    .then(() => waitTillNone('namespace', undefined, name))
                    .catch(err => {
                    if (!errOk) {
                        return Common.oops(this)(err);
                    }
                });
            });
        };
        const createIt = (name) => {
            it(`should create namespace ${name} via ${kubectl}`, () => {
                return CLI.command(`${kubectl} create namespace ${name}`, this.app)
                    .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME(name) }))
                    .then(selector => waitForGreen(this.app, selector))
                    .catch(Common.oops(this));
            });
        };
        const createPod = (ns) => {
            it(`should create sample pod in namespace ${ns} from URL via ${kubectl}`, () => {
                return CLI.command(`${kubectl} create -f https://raw.githubusercontent.com/kubernetes/examples/master/staging/pod -n ${ns}`, this.app)
                    .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME('nginx') }))
                    .then(selector => waitForGreen(this.app, selector))
                    .catch(Common.oops(this));
            });
            it(`should show the sample pod in namespace ${ns} in sidecar via ${kubectl}`, () => {
                return CLI.command(`${kubectl} get pod nginx -n ${ns} -o yaml`, this.app)
                    .then(ReplExpect.justOK)
                    .then(SidecarExpect.open)
                    .then(SidecarExpect.showing('nginx', undefined, undefined, ns))
                    .catch(Common.oops(this));
            });
        };
        const defaultFilepath = expandHomeDir('~/.kube/config');
        const getKUBECONFIGFilepath = () => {
            if (process.env.KUBECONFIG) {
                const kconfigEnv = process.env.KUBECONFIG.slice(0);
                return kconfigEnv.split(/:/)[0];
            }
            else {
                return defaultFilepath;
            }
        };
        const getKUBECONFIG = () => {
            if (process.env.KUBECONFIG) {
                return readFileSync(getKUBECONFIGFilepath());
            }
            else {
                return execSync('kubectl config view');
            }
        };
        const addNamespaceToKUBECONFIG = (ns, contextName) => {
            it('should add a new context', () => __awaiter(this, void 0, void 0, function* () {
                try {
                    const kconfig = parseYAML(getKUBECONFIG().toString());
                    const newOnesFilepath = path.join(path.dirname(getKUBECONFIGFilepath()), 'forTesting.yml');
                    kconfig.contexts[0].context.namespace = ns;
                    kconfig.contexts[0].name = contextName;
                    writeFileSync(newOnesFilepath, safeDump(kconfig));
                    yield this.app.client.execute((defaultFilepath, newOnesFilepath) => {
                        process.env.KUBECONFIG = `${process.env.KUBECONFIG || defaultFilepath}:${newOnesFilepath}`;
                    }, defaultFilepath, newOnesFilepath);
                }
                catch (err) {
                    return Common.oops(this)(err);
                }
            }));
        };
        const listContextsAndExpectDefault = () => {
            it('should list contexts and show the default context', () => __awaiter(this, void 0, void 0, function* () {
                try {
                    const currentContext = yield CLI.command(`context`, this.app)
                        .then(ReplExpect.okWithCustom({ selector: ' ' }))
                        .then(selector => this.app.client.getText(selector));
                    const currentContextAsIndicatedByContextsTable = yield CLI.command(`contexts`, this.app)
                        .then(ReplExpect.okWithCustom({
                        selector: `.selected-row .entity-name[data-key="NAME"]`
                    }))
                        .then(selector => this.app.client.getText(selector));
                    assert.strictEqual(currentContextAsIndicatedByContextsTable, currentContext);
                }
                catch (err) {
                    return Common.oops(this)(err);
                }
            }));
        };
        const listContextsAndExpectGiven = (contextName) => {
            it(`should list contexts and show the context ${contextName}`, () => __awaiter(this, void 0, void 0, function* () {
                try {
                    const allContextNames = yield CLI.command(`contexts`, this.app)
                        .then(ReplExpect.okWithCustom({ selector: ' ' }))
                        .then(selector => this.app.client.elements(`${selector} .entity-name[data-key="NAME"]`))
                        .then(elements => elements.value.map(_ => _.ELEMENT))
                        .then(elements => Promise.all(elements.map(element => this.app.client.elementIdText(element))))
                        .then(texts => texts.map(_ => _.value));
                    assert.ok(allContextNames.find(_ => _ === contextName));
                }
                catch (err) {
                    return Common.oops(this)(err);
                }
            }));
        };
        const listPodsAndExpectNone = (ns) => {
            it('should list pods and show nothing', () => {
                return CLI.command(`${kubectl} get pods -n ${ns}`, this.app).then(ReplExpect.error(404));
            });
        };
        const listPodsAndExpectOne = (name, ns) => {
            it(`should list pods and show ${name} maybe in namespace ${ns || 'nope'}`, () => {
                return CLI.command(`${kubectl} get pods ${ns ? '-n ' + ns : ''}`, this.app)
                    .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME(name) }))
                    .then(selector => waitForGreen(this.app, selector))
                    .catch(Common.oops(this));
            });
        };
        const switchToContext = (contextName) => {
            it(`should switch to the context ${contextName}`, () => __awaiter(this, void 0, void 0, function* () {
                try {
                    const selector = yield CLI.command(`contexts`, this.app).then(ReplExpect.okWithCustom({
                        selector: Selectors.BY_NAME(contextName)
                    }));
                    yield this.app.client.click(`${selector} .entity-name.clickable`);
                    yield this.app.client.waitForExist(`${selector} .selected-row`);
                    const selector2 = yield CLI.command(`contexts`, this.app).then(ReplExpect.okWithCustom({
                        selector: Selectors.BY_NAME(contextName)
                    }));
                    yield this.app.client.waitForExist(`${selector2} .selected-row`);
                }
                catch (err) {
                    return Common.oops(this)(err);
                }
            }));
        };
        const ns = createNS();
        listContextsAndExpectDefault();
        createIt(ns);
        addNamespaceToKUBECONFIG(ns, 'holla');
        listContextsAndExpectGiven('holla');
        listPodsAndExpectNone(ns);
        createPod(ns);
        listPodsAndExpectOne('nginx', ns);
        switchToContext('holla');
        listPodsAndExpectOne('nginx');
        deleteIt(ns);
    });
});
//# sourceMappingURL=contexts.js.map