/*
 * Copyright 2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
// FIXME:
/* eslint-disable react/prop-types */
import Debug from 'debug';
import * as React from 'react';
import { i18n, inBrowser, eventChannelUnsafe, findThemeByName, getPersistedThemeChoice, getDefaultTheme } from '@kui-shell/core';
import Card from '../spi/Card';
import KuiContext from './context';
import { ComboSidecar, InputStripe, StatusStripe, TabContainer, Loading, Alert } from '../..';
import KuiIcon from '../../../icons/png/WelcomeLight.png';
const debug = Debug('<Kui/>');
const strings = i18n('client');
const Popup = React.lazy(() => import(/* webpackMode: "lazy" */ './Popup'));
const defaultThemeProperties = {
    components: 'carbon',
    topTabNames: 'fixed'
};
/**
 * Render the main body of our client.
 *
 * |Notes on Session Initialization|: to provide custom views for
 * session initialization (only relevant for browser-based hosted
 * Kui), you can instantiate <Kui/> with these properties (defined in
 * KuiConfiguration), show here with some sample views:
 *
 * <Kui
 *    loading={<div className="kui--hero-text">Hold on...</div>}
 *    reinit={<div className="kui--hero-text">Connection broken...</div>}
 *    loadingError={err => <div className="kui--hero-text">{err.toString()}</div>}
 *    loadingDone={<div>Welcome to Kui</div>}
 * />
 *
 */
export class Kui extends React.PureComponent {
    constructor(props) {
        super(props);
        eventChannelUnsafe.on('/theme/change', this.onThemeChange.bind(this));
        setTimeout(() => __awaiter(this, void 0, void 0, function* () {
            const { theme } = yield findThemeByName((yield getPersistedThemeChoice()) || (yield getDefaultTheme()));
            this.setState(curState => {
                const stateWithThemeProps = Object.assign({}, theme, curState);
                debug('state with theme props', stateWithThemeProps);
                return stateWithThemeProps;
            });
        }));
        if (!props.noBootstrap) {
            import('@kui-shell/core')
                .then(_ => _.bootIntoSandbox())
                .then(() => {
                this.setState({ isBootstrapped: true });
            });
        }
        try {
            this.state = Object.assign({}, this.defaultSessionBehavior(), this.defaultFeatureFlag(), props, {
                isBootstrapped: !!props.noBootstrap
            });
            debug('initial state:inBrowser?', inBrowser());
            debug('initial state:given properties', props);
            debug('initial state:final value', this.state);
        }
        catch (err) {
            console.log('using default configuration');
            this.state = {
                isBootstrapped: !!props.noBootstrap
            };
        }
    }
    defaultFeatureFlag() {
        return {
            sidecarName: 'breadcrumb',
            showWelcomeMax: -1
        };
    }
    defaultLoading() {
        return (React.createElement(Loading, { className: "somewhat-larger-text", description: strings('Please wait while we connect to your cluster') }));
    }
    defaultReinit() {
        return (React.createElement(Alert, { hideCloseButton: true, alert: {
                type: 'error',
                title: strings('Lost connection to your cluster'),
                body: strings('Attempting to reconnect...')
            } }));
    }
    defaultLoadingDone() {
        return (repl) => !inBrowser() ? (undefined) : (React.createElement(Card, { titleInHeader: true, bodyInHeader: true, title: strings('Successfully connected to your cluster'), repl: repl, icon: KuiIcon }, strings('loadingDone:content')));
    }
    defaultLoadingError() {
        return err => (React.createElement(Alert, { alert: { type: 'error', title: strings('Error connecting to your cluster'), body: err.toString() } }));
    }
    /**
     * For browser-based clients, this defines the default UI for
     * session initialization.
     *
     */
    defaultSessionBehavior() {
        const behavior = !inBrowser()
            ? {}
            : {
                loading: this.defaultLoading(),
                reinit: this.defaultReinit(),
                loadingError: this.defaultLoadingError(),
                loadingDone: this.defaultLoadingDone()
            };
        debug('defaultSessionBehavior', behavior);
        return behavior;
    }
    onThemeChange({ themeModel }) {
        this.setState(curState => {
            // note the priority order, from highest to lowest:
            //  1) any properties defined by the theme (since we just switched themes)
            //  2) any properties defined by the container of this <Kui/>
            //  3) default choices
            //  4) any prior state
            // re: ordering of 3 and 4, see https://github.com/IBM/kui/issues/4423
            const stateAfterThemeChange = Object.assign({}, curState, defaultThemeProperties, this.props, themeModel);
            debug('state after theme change', stateAfterThemeChange);
            return stateAfterThemeChange;
        });
    }
    componentDidCatch(error, errorInfo) {
        console.error(error, errorInfo);
    }
    render() {
        if (!this.state.isBootstrapped) {
            return React.createElement(Loading, null);
        }
        if (this.props.isPopup && this.props.commandLine) {
            return (React.createElement(KuiContext.Provider, { value: this.state },
                React.createElement(React.Suspense, { fallback: React.createElement("div", null) },
                    React.createElement(Popup, { commandLine: this.props.commandLine }, this.props.children))));
        }
        else {
            const bottom = !!this.props.bottomInput && React.createElement(InputStripe, null, this.props.bottomInput);
            return (React.createElement(KuiContext.Provider, { value: this.state },
                React.createElement("div", { className: "kui--full-height" },
                    React.createElement(TabContainer, { noActiveInput: !!this.props.bottomInput, bottom: bottom },
                        React.createElement(ComboSidecar, null)),
                    this.props.toplevel,
                    React.createElement(StatusStripe, null, this.props.children))));
        }
    }
}
export default Kui;
//# sourceMappingURL=Kui.js.map