/*
 * Copyright 2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* eslint-disable @typescript-eslint/no-empty-function */
import * as React from 'react';
import { eventBus } from '@kui-shell/core';
import { Loading } from '../../..';
import { getStateFromMMR } from './TopNavSidecar';
import { BaseSidecar, cwd } from './BaseSidecar';
import Navigation from '../../spi/Navigation';
/** Lazily load KuiContent; see https://github.com/IBM/kui/issues/3746 */
const KuiContent = React.lazy(() => import('../../Content/KuiContent'));
/**
 *
 * LeftNavSidecar
 * -------------------------
 * | <TitleBar/>           |
 * -------------------------
 * | A1   |                |
 * |  a1  | <Content>      |
 * |  a2  |  <KuiContent/> |
 * | B1   | </Content>     |
 * |  b1  |                |
 * |  b2  |                |
 * -------------------------
 *  ^^^^^ <SideNav/>
 *   A1, B1: <SideNavMenu/>
 *   a1, b1: <SideNavMenuItem/>
 *
 */
export default class LeftNavSidecar extends BaseSidecar {
    constructor(props) {
        super(props);
        const onResponse = this.onResponse.bind(this);
        eventBus.onNavResponse(this.props.uuid, onResponse);
        this.cleaners.push(() => eventBus.offNavResponse(this.props.uuid, onResponse));
        this.state = {
            repl: undefined,
            tab: undefined,
            dom: undefined,
            history: undefined,
            current: undefined
        };
    }
    /** Is getState() idempotent? i.e. Will two command executions that satisfy `sameCommand` always produce the same response? */
    idempotent() {
        return true;
    }
    /** Should we display Back/Forward arrows for history navigation? */
    useArrowNavigation() {
        return false;
    }
    /** 30/70 split between the Terminal and the LeftNavSidecar */
    defaultWidth() {
        return "70%" /* Split70 */;
    }
    /** @return a `HistoryEntry` for the given `Response` */
    getState(tab, response, execUUID, argvNoOptions, parsedOptions) {
        const navigations = [];
        // get state from each of the left navigation
        response.menus.forEach(menu => {
            const state = getStateFromMMR(tab, { modes: menu.items }, execUUID, argvNoOptions, parsedOptions);
            navigations.push(Object.assign({ title: menu.label }, state));
        });
        return {
            cwd: cwd(),
            execUUID,
            argvNoOptions,
            parsedOptions,
            allNavs: navigations,
            allLinks: response.links || [],
            current: { menuIdx: 0, tabIdx: navigations[0].currentTabIndex },
            response
        };
    }
    isFixedWidth() {
        return true;
    }
    changeCurrent(menuIdx, tabIdx) {
        this.setState(({ current, history }) => {
            const newCurrent = Object.assign({}, current, { current: { menuIdx, tabIdx } });
            history.updateActive(newCurrent);
            return {
                current: newCurrent
            };
        });
    }
    /** render the leftnav part */
    nav() {
        return React.createElement(Navigation, { tab: this.state.tab, current: this.current, changeCurrent: this.changeCurrent.bind(this) });
    }
    bodyContent(tabIdx, menuIdx = 0) {
        return (React.createElement(React.Suspense, { fallback: React.createElement("div", null) },
            React.createElement(KuiContent, { key: `${this.current.execUUID}-${menuIdx}-${tabIdx}`, tab: this.state.tab, mode: this.current.allNavs[menuIdx].tabs[tabIdx], isActive: true, args: { argvNoOptions: this.state.current.argvNoOptions, parsedOptions: this.state.current.parsedOptions }, response: { modes: this.current.response.menus[menuIdx].items } })));
    }
    bodyContainer(tabIdx, menuIdx) {
        return React.createElement("div", { className: "kui--sidecar-content" }, this.bodyContent(tabIdx, menuIdx));
    }
    render() {
        if (!this.current || !this.current.response) {
            return React.createElement(Loading, null);
        }
        return (React.createElement("div", { className: 'kui--sidecar kui--inverted-color-context kui--nav-view ' + this.width(), ref: dom => this.setState({ dom }), "data-view": "leftnav" },
            ' ',
            this.title({ breadcrumbs: this.current.response.breadcrumbs }),
            React.createElement("div", { className: "kui--sidecar-header-and-body zoomable" },
                this.nav(),
                this.bodyContainer(this.current.current.tabIdx, this.current.current.menuIdx))));
    }
}
//# sourceMappingURL=LeftNavSidecar.js.map