"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isError = isError;
exports.isProcessing = isProcessing;
exports.isActive = isActive;
exports.isCancelled = isCancelled;
exports.isEmpty = isEmpty;
exports.isOk = isOk;
exports.isOops = isOops;
exports.isFinished = isFinished;
exports.hasCommand = hasCommand;
exports.isAnnouncement = isAnnouncement;
exports.hasUUID = hasUUID;
exports.hasValue = hasValue;
exports.Active = Active;
exports.Announcement = Announcement;
exports.Processing = Processing;
exports.Empty = Empty;
exports.Cancelled = Cancelled;
exports.Finished = Finished;

var _path = require("path");

var _core = require("@kui-shell/core");

/*
 * Copyright 2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/** Capture the current working directory */
function cwd() {
  const dir = (0, _core.inBrowser)() ? process.env.PWD : process.cwd();
  return dir === process.env.HOME ? '~' : (0, _path.basename)(dir);
}

function isError(response) {
  return response.constructor === Error || response.constructor === _core.UsageError;
}

function isProcessing(block) {
  return block.state === "processing"
  /* Processing */
  ;
}

function isActive(block) {
  return block.state === "repl-active"
  /* Active */
  ;
}

function isCancelled(block) {
  return block.state === "cancelled"
  /* Cancelled */
  ;
}

function isEmpty(block) {
  return block.state === "empty"
  /* Empty */
  ;
}

function isOk(block) {
  return block.state === "valid-response"
  /* ValidResponse */
  ;
}

function isOops(block) {
  return block.state === "error"
  /* Error */
  ;
}

function isFinished(block) {
  return isOops(block) || isCancelled(block) || isOk(block) || isEmpty(block);
}

function hasCommand(block) {
  return !isActive(block) && !isEmpty(block);
}

function isAnnouncement(block) {
  const blockModel = block;
  return blockModel.state === "valid-response"
  /* ValidResponse */
  && blockModel.isAnnouncement === true;
}

function hasUUID(block) {
  return !isActive(block) && !isEmpty(block) && !isAnnouncement(block);
}

function hasValue(block) {
  return typeof block.value === 'string';
}
/** Transform to Active */


function Active(initialValue) {
  return {
    cwd: cwd(),
    state: "repl-active"
    /* Active */
    ,
    value: initialValue
  };
}
/** Transform to AnnouncementBlock */


function Announcement(response) {
  return {
    response,
    isAnnouncement: true,
    startTime: new Date(),
    cwd: cwd(),
    state: "valid-response"
    /* ValidResponse */

  };
}
/** Transform to Processing */


function Processing(block, command, execUUID) {
  return {
    command,
    cwd: block.cwd,
    execUUID: execUUID,
    startTime: new Date(),
    state: "processing"
    /* Processing */

  };
}
/** Transform to Empty */


function Empty(block) {
  return {
    cwd: block.cwd,
    state: "empty"
    /* Empty */

  };
}
/** Transform to Cancelled */


function Cancelled(block) {
  if (isProcessing(block)) {
    return {
      cwd: block.cwd,
      command: block.command,
      execUUID: block.execUUID,
      startTime: block.startTime,
      state: "cancelled"
      /* Cancelled */

    };
  } else {
    return Empty(block);
  }
}
/** Transform to Finished */


function Finished(block, response, cancelled) {
  if (cancelled) {
    return Cancelled(block);
  } else if (isError(response)) {
    return {
      response,
      cwd: block.cwd,
      command: block.command,
      state: "error"
      /* Error */
      ,
      execUUID: block.execUUID,
      startTime: block.startTime
    };
  } else {
    return {
      response,
      cwd: block.cwd,
      command: block.command,
      execUUID: block.execUUID,
      startTime: block.startTime,
      state: "valid-response"
      /* ValidResponse */

    };
  }
}