/*
 * Copyright 2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { basename } from 'path';
import { UsageError, inBrowser } from '@kui-shell/core';
/** Capture the current working directory */
function cwd() {
    const dir = inBrowser() ? process.env.PWD : process.cwd();
    return dir === process.env.HOME ? '~' : basename(dir);
}
export function isError(response) {
    return response.constructor === Error || response.constructor === UsageError;
}
export function isProcessing(block) {
    return block.state === "processing" /* Processing */;
}
export function isActive(block) {
    return block.state === "repl-active" /* Active */;
}
export function isCancelled(block) {
    return block.state === "cancelled" /* Cancelled */;
}
export function isEmpty(block) {
    return block.state === "empty" /* Empty */;
}
export function isOk(block) {
    return block.state === "valid-response" /* ValidResponse */;
}
export function isOops(block) {
    return block.state === "error" /* Error */;
}
export function isFinished(block) {
    return isOops(block) || isCancelled(block) || isOk(block) || isEmpty(block);
}
export function hasCommand(block) {
    return !isActive(block) && !isEmpty(block);
}
export function isAnnouncement(block) {
    const blockModel = block;
    return blockModel.state === "valid-response" /* ValidResponse */ && blockModel.isAnnouncement === true;
}
export function hasUUID(block) {
    return !isActive(block) && !isEmpty(block) && !isAnnouncement(block);
}
export function hasValue(block) {
    return typeof block.value === 'string';
}
/** Transform to Active */
export function Active(initialValue) {
    return {
        cwd: cwd(),
        state: "repl-active" /* Active */,
        value: initialValue
    };
}
/** Transform to AnnouncementBlock */
export function Announcement(response) {
    return {
        response,
        isAnnouncement: true,
        startTime: new Date(),
        cwd: cwd(),
        state: "valid-response" /* ValidResponse */
    };
}
/** Transform to Processing */
export function Processing(block, command, execUUID) {
    return {
        command,
        cwd: block.cwd,
        execUUID: execUUID,
        startTime: new Date(),
        state: "processing" /* Processing */
    };
}
/** Transform to Empty */
export function Empty(block) {
    return {
        cwd: block.cwd,
        state: "empty" /* Empty */
    };
}
/** Transform to Cancelled */
export function Cancelled(block) {
    if (isProcessing(block)) {
        return {
            cwd: block.cwd,
            command: block.command,
            execUUID: block.execUUID,
            startTime: block.startTime,
            state: "cancelled" /* Cancelled */
        };
    }
    else {
        return Empty(block);
    }
}
/** Transform to Finished */
export function Finished(block, response, cancelled) {
    if (cancelled) {
        return Cancelled(block);
    }
    else if (isError(response)) {
        return {
            response,
            cwd: block.cwd,
            command: block.command,
            state: "error" /* Error */,
            execUUID: block.execUUID,
            startTime: block.startTime
        };
    }
    else {
        return {
            response,
            cwd: block.cwd,
            command: block.command,
            execUUID: block.execUUID,
            startTime: block.startTime,
            state: "valid-response" /* ValidResponse */
        };
    }
}
//# sourceMappingURL=BlockModel.js.map