"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.preload = preload;
exports.plugin = plugin;

var _fs = require("fs");

var _path = require("path");

var _core = require("@kui-shell/core");

/*
 * Copyright 2017-20 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};

/**
 * Is the given filepath a directory?
 *
 */
const isDirectory = filepath => new Promise((resolve, reject) => {
  (0, _fs.lstat)(filepath, (err, stats) => {
    if (err) {
      reject(err);
    } else {
      if (stats.isSymbolicLink()) {
        // debug('following symlink')
        // TODO: consider turning these into the better async calls?
        return (0, _fs.realpath)(filepath, (err, realpath) => {
          if (err) {
            reject(err);
          } else {
            isDirectory(realpath).then(resolve).catch(reject);
          }
        });
      }

      resolve(stats.isDirectory());
    }
  });
}).catch(err => {
  if (err.code === 'ENOENT') {
    return false;
  } else {
    throw err;
  }
});
/**
 * Tab completion handler for local files
 *
 */


function completeLocalFiles(tab, commandLine, {
  toBeCompleted
}) {
  return __awaiter(this, void 0, void 0, function* () {
    return (yield tab.REPL.rexec(`fscomplete -- "${toBeCompleted}"`)).content;
  });
}

function doComplete(args) {
  const last = args.command.substring(args.command.indexOf('-- ') + '-- '.length).replace(/^"(.*)"$/, '$1'); // dirname will "foo" in the above example; it
  // could also be that last is itself the name
  // of a directory

  const lastIsDir = last.charAt(last.length - 1) === '/';
  const dirname = lastIsDir ? last : (0, _path.dirname)(last); // debug('suggest local file', dirname, last)

  if (dirname) {
    // then dirname exists! now scan the directory so we can find matches
    return new Promise((resolve, reject) => {
      const dirToScan = (0, _core.expandHomeDir)(dirname);
      (0, _fs.readdir)(dirToScan, (err, files) => __awaiter(this, void 0, void 0, function* () {
        if (err) {
          console.error('fs.readdir error', err);
          reject(err);
        } else {
          const partial = (0, _path.basename)(last) + (lastIsDir ? '/' : '');
          const partialHasADot = partial.startsWith('.');
          const matches = files.filter(_f => {
            const f = _f; // exclude dot files from tab completion, also emacs ~ temp files just for fun

            return (lastIsDir || f.indexOf(partial) === 0) && !f.endsWith('~') && f !== '.' && f !== '..' && (partialHasADot || !f.startsWith('.'));
          }); // add a trailing slash to any matched directory names

          const lastHasPath = /\//.test(last);
          resolve({
            mode: 'raw',
            content: yield Promise.all(matches.map(match => __awaiter(this, void 0, void 0, function* () {
              const completion = lastIsDir ? match : match.substring(partial.length); // show a special label only if we have a dirname prefix

              const label = lastHasPath ? (0, _path.basename)(match) : undefined;

              if (yield isDirectory((0, _path.join)(dirToScan, match))) {
                return {
                  completion: `${completion}/`,
                  label: label ? `${label}/` : undefined
                };
              } else {
                return {
                  completion,
                  addSpace: true,
                  label
                };
              }
            })))
          });
        }
      }));
    });
  }
}
/**
 * Entry point to register tab completion handlers. Register this as a
 * very low priority enumerator. We don't want e.g. to enumerate files
 * when the user is doing a `git checkout myBran<tab>`.
 *
 */


function preload() {
  (0, _core.registerTabCompletionEnumerator)(completeLocalFiles, -100);
}

function plugin(registrar) {
  registrar.listen('/fscomplete', doComplete, {
    hidden: true,
    requiresLocal: true
  });
}