/**
 * Copyright IBM Corp. 2020
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import React from 'react';
import Menu, { MenuItem, MenuRadioGroup, MenuSelectableItem, MenuDivider } from '../Menu';
import { mount } from 'enzyme';
import { settings } from 'carbon-components';
import { describe, expect } from 'window-or-global';
var prefix = settings.prefix;
describe('Menu', function () {
  describe('renders as expected', function () {
    describe('menu', function () {
      it('receives the expected classes when closed', function () {
        var wrapper = mount( /*#__PURE__*/React.createElement(Menu, null));
        var container = wrapper.childAt(0).childAt(0);
        expect(container.hasClass("".concat(prefix, "--menu"))).toBe(true);
        expect(container.hasClass("".concat(prefix, "--menu--open"))).toBe(false);
      });
      it('receives the expected classes when opened', function () {
        var wrapper = mount( /*#__PURE__*/React.createElement(Menu, {
          open: true
        }));
        var container = wrapper.childAt(0).childAt(0);
        expect(container.hasClass("".concat(prefix, "--menu"))).toBe(true);
        expect(container.hasClass("".concat(prefix, "--menu--open"))).toBe(true);
      });
    });
    describe('option', function () {
      it('receives the expected classes', function () {
        var wrapper = mount( /*#__PURE__*/React.createElement(MenuItem, {
          label: "Copy"
        }));
        var container = wrapper.childAt(0).childAt(0);
        expect(container.hasClass("".concat(prefix, "--menu-option"))).toBe(true);
      });
      it('renders props.label', function () {
        var wrapper = mount( /*#__PURE__*/React.createElement(MenuItem, {
          label: "Copy"
        }));
        expect(wrapper.find("span.".concat(prefix, "--menu-option__label")).text()).toBe('Copy');
        expect(wrapper.find("span.".concat(prefix, "--menu-option__label")).prop('title')).toBe('Copy');
      });
      it('renders props.shortcut when provided', function () {
        var wrapper = mount( /*#__PURE__*/React.createElement(MenuItem, {
          label: "Copy",
          shortcut: "\u2318C"
        }));
        expect(wrapper.find("div.".concat(prefix, "--menu-option__info")).length).toBeGreaterThan(0);
        expect(wrapper.find("div.".concat(prefix, "--menu-option__info")).text()).toBe('⌘C');
      });
      it('respects props.disabled', function () {
        var wrapper = mount( /*#__PURE__*/React.createElement(MenuItem, {
          label: "Copy",
          disabled: true
        }));
        var content = wrapper.find("div.".concat(prefix, "--menu-option__content"));
        expect(content.hasClass("".concat(prefix, "--menu-option__content--disabled"))).toBe(true);
        expect(wrapper.find("li.".concat(prefix, "--menu-option")).prop('aria-disabled')).toBe(true);
      });
      it('supports danger kind', function () {
        var wrapper = mount( /*#__PURE__*/React.createElement(MenuItem, {
          label: "Delete",
          kind: "danger"
        }));
        var option = wrapper.find(".".concat(prefix, "--menu-option"));
        expect(option.hasClass("".concat(prefix, "--menu-option--danger"))).toBe(true);
      });
      it('renders props.children as submenu', function () {
        var wrapper = mount( /*#__PURE__*/React.createElement(Menu, null, /*#__PURE__*/React.createElement(MenuItem, {
          label: "Format"
        }, /*#__PURE__*/React.createElement(MenuItem, {
          label: "Bold"
        }), /*#__PURE__*/React.createElement(MenuItem, {
          label: "Italic"
        }))));
        var level1 = wrapper.find("li.".concat(prefix, "--menu-option")).at(0);
        expect(level1.find("ul.".concat(prefix, "--menu")).length).toBeGreaterThan(0);
      });
    });
    describe('radiogroup', function () {
      it('children have role "menuitemradio"', function () {
        var wrapper = mount( /*#__PURE__*/React.createElement(MenuRadioGroup, {
          label: "Share with",
          items: ['None', 'All']
        }));
        var options = wrapper.find("li.".concat(prefix, "--menu-option"));
        expect(options.every('li[role="menuitemradio"]')).toBe(true);
      });
    });
    describe('selectable', function () {
      it('has role "menuitemcheckbox"', function () {
        var wrapper = mount( /*#__PURE__*/React.createElement(MenuSelectableItem, {
          label: "Publish"
        }));
        var container = wrapper.childAt(0);
        expect(container.prop('role')).toBe('menuitemcheckbox');
      });
    });
    describe('divider', function () {
      it('receives the expected classes', function () {
        var wrapper = mount( /*#__PURE__*/React.createElement(MenuDivider, null));
        var container = wrapper.childAt(0);
        expect(container.hasClass("".concat(prefix, "--menu-divider"))).toBe(true);
      });
      it('has role "separator"', function () {
        var wrapper = mount( /*#__PURE__*/React.createElement(MenuDivider, null));
        var container = wrapper.childAt(0);
        expect(container.prop('role')).toBe('separator');
      });
    });
  });
});