"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resetFocus = resetFocus;
exports.focusNode = focusNode;
exports.getValidNodes = getValidNodes;
exports.getNextNode = getNextNode;
exports.getFirstSubNode = getFirstSubNode;
exports.getParentNode = getParentNode;
exports.getParentMenu = getParentMenu;
exports.clickedElementHasSubnodes = clickedElementHasSubnodes;
exports.capWithinRange = capWithinRange;
exports.getPosition = getPosition;

var _carbonComponents = require("carbon-components");

var prefix = _carbonComponents.settings.prefix;

function resetFocus(el) {
  if (el) {
    var _el$querySelectorAll;

    Array.from((_el$querySelectorAll = el.querySelectorAll('[tabindex="0"]')) !== null && _el$querySelectorAll !== void 0 ? _el$querySelectorAll : []).forEach(function (node) {
      node.tabIndex = -1;
    });
  }
}

function focusNode(node) {
  if (node) {
    node.tabIndex = 0;
    node.focus();
  }
}

function getValidNodes(list) {
  var level = list.dataset.level;
  var nodes = [];

  if (level) {
    var submenus = Array.from(list.querySelectorAll('[data-level]'));
    nodes = Array.from(list.querySelectorAll("li.".concat(prefix, "--menu-option"))).filter(function (child) {
      return !submenus.some(function (submenu) {
        return submenu.contains(child);
      });
    });
  }

  return nodes.filter(function (node) {
    return node.matches(":not(.".concat(prefix, "--menu-option--disabled)"));
  });
}

function getNextNode(current, direction) {
  var menu = getParentMenu(current);
  var nodes = getValidNodes(menu);
  var currentIndex = nodes.indexOf(current);
  var nextNode = nodes[currentIndex + direction];
  return nextNode || null;
}

function getFirstSubNode(node) {
  var submenu = node.querySelector("ul.".concat(prefix, "--menu"));

  if (submenu) {
    var subnodes = getValidNodes(submenu);
    return subnodes[0] || null;
  }

  return null;
}

function getParentNode(node) {
  if (node) {
    var parentNode = node.parentNode.closest("li.".concat(prefix, "--menu-option"));
    return parentNode || null;
  }

  return null;
}

function getParentMenu(el) {
  if (el) {
    var parentMenu = el.parentNode.closest("ul.".concat(prefix, "--menu"));
    return parentMenu || null;
  }

  return null;
}

function clickedElementHasSubnodes(e) {
  if (e) {
    var closestFocusableElement = e.target.closest('[tabindex]');

    if ((closestFocusableElement === null || closestFocusableElement === void 0 ? void 0 : closestFocusableElement.tagName) === 'LI') {
      return getFirstSubNode(closestFocusableElement) !== null;
    }
  }

  return false;
}
/**
 * @param {number} [value] The value to cap
 * @param {number} [min] The minimum of the range
 * @param {number} [max] The maximum of the range
 * @returns {number} Whether or not the element fits inside the boundaries on the given axis
 */


function capWithinRange(value, min, max) {
  if (value > max) {
    return max;
  }

  if (value < min) {
    return min;
  }

  return value;
}
/**
 * @param {number[]} [elementDimensions] The dimensions of the element: [width, height]
 * @param {number[]} [position] The desired position of the element: [x, y]
 * @param {number[]} [boundaries] The boundaries of the container the element should be contained in: [minX, minY, maxX, maxY]
 * @param {string} [axis="x"] Which axis to check. Either "x" or "y"
 * @returns {boolean} Whether or not the element fits inside the boundaries on the given axis
 */


function elementFits(elementDimensions, position, boundaries) {
  var axis = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 'x';
  var index = axis === 'y' ? 1 : 0;
  var min = boundaries[index];
  var max = boundaries[index + 2];
  var start = position[index];
  var end = position[index] + elementDimensions[index];
  return start >= min && end <= max;
}
/**
 * @param {number[]} [elementDimensions] The dimensions of the element: [width, height]
 * @param {number[]} [targetBoundaries] The boundaries of the target the element should attach to: [minX, minY, maxX, maxY]
 * @param {number[]} [containerBoundaries] The boundaries of the container the element should be contained in: [minX, minY, maxX, maxY]
 * @param {number} [preferredDirection=1] Which direction is preferred. Either 1 (right right) or -1 (to left)
 * @returns {object} The determined position and direction of the elemnt: { position: [x, y], direction: 1 | -1 }
 */


function getPosition(elementDimensions, targetBoundaries, containerBoundaries) {
  var preferredDirection = arguments.length > 3 && arguments[3] !== undefined ? arguments[3] : 1;
  var position = [0, 0];
  var direction = preferredDirection; // x

  position[0] = direction === 1 ? targetBoundaries[0] : targetBoundaries[2] - elementDimensions[0];
  var xFits = elementFits(elementDimensions, position, containerBoundaries, 'x');

  if (!xFits) {
    direction = direction * -1;
    position[0] = direction === 1 ? targetBoundaries[0] : targetBoundaries[2] - elementDimensions[0];
  } // y


  position[1] = targetBoundaries[3];
  var yFits = elementFits(elementDimensions, position, containerBoundaries, 'y');

  if (!yFits) {
    position[1] = targetBoundaries[1] - elementDimensions[1];
  }

  return {
    position: position,
    direction: direction
  };
}