/**
 * Copyright IBM Corp. 2015, 2020
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

const enabled$1 = {};

try {
  if (process.env.CARBON_ENABLE_CSS_CUSTOM_PROPERTIES) {
    if (process.env.CARBON_ENABLE_CSS_CUSTOM_PROPERTIES === 'true') {
      enabled$1.enableCssCustomProperties = true;
    } else {
      enabled$1.enableCssCustomProperties = false;
    }
  } else {
    enabled$1.enableCssCustomProperties = false;
  }

  if (process.env.CARBON_ENABLE_USE_CONTROLLED_STATE_WITH_VALUE) {
    if (process.env.CARBON_ENABLE_USE_CONTROLLED_STATE_WITH_VALUE === 'true') {
      enabled$1.enableUseControlledStateWithValue = true;
    } else {
      enabled$1.enableUseControlledStateWithValue = false;
    }
  } else {
    enabled$1.enableUseControlledStateWithValue = false;
  }

  if (process.env.CARBON_ENABLE_CSS_GRID) {
    if (process.env.CARBON_ENABLE_CSS_GRID === 'true') {
      enabled$1.enableCssGrid = true;
    } else {
      enabled$1.enableCssGrid = false;
    }
  } else {
    enabled$1.enableCssGrid = false;
  }

  if (process.env.CARBON_ENABLE_V11_RELEASE) {
    if (process.env.CARBON_ENABLE_V11_RELEASE === 'true') {
      enabled$1.enableV11Release = true;
    } else {
      enabled$1.enableV11Release = false;
    }
  } else {
    enabled$1.enableV11Release = false;
  }
} catch (error) {
  enabled$1.enableCssCustomProperties = false;
  enabled$1.enableUseControlledStateWithValue = false;
  enabled$1.enableCssGrid = false;
  enabled$1.enableV11Release = false;
}

const featureFlagInfo = [{
  name: "enable-css-custom-properties",
  description: "Describe what the flag does",
  enabled: enabled$1.enableCssCustomProperties
}, {
  name: "enable-use-controlled-state-with-value",
  description: "Enable components to be created in either a controlled or uncontrolled mode\n",
  enabled: enabled$1.enableUseControlledStateWithValue
}, {
  name: "enable-css-grid",
  description: "Enable CSS Grid Layout in the Grid and Column React components\n",
  enabled: enabled$1.enableCssGrid
}, {
  name: "enable-v11-release",
  description: "Enable the features and functionality for the v11 Release\n",
  enabled: enabled$1.enableV11Release
}];

class FeatureFlagScope {
  constructor(flags) {
    this.flags = new Map();

    if (flags) {
      Object.keys(flags).forEach((key) => {
        this.flags.set(key, flags[key]);
      });
    }
  }

  /**
   * Check to see if a flag exists
   * @param {string} name
   */
  checkForFlag(name) {
    if (!this.flags.has(name)) {
      throw new Error(
        `Unable to find a feature flag with the name: \`${name}\``
      );
    }
  }

  /**
   * Add a feature flag
   * @param {string} name
   * @param {boolean} enabled
   */
  add(name, enabled) {
    if (this.flags.has(name)) {
      throw new Error(`The feature flag: ${name} already exists`);
    }
    this.flags.set(name, enabled);
  }

  /**
   * Enable a feature flag
   * @param {string} name
   */
  enable(name) {
    this.checkForFlag(name);
    this.flags.set(name, true);
  }

  /**
   * Disable a feature flag
   * @param {string} name
   */
  disable(name) {
    this.checkForFlag(name);
    this.flags.set(name, false);
  }

  /**
   * Merge the given feature flags with the current set of feature flags.
   * Duplicate keys will be set to the value in the given feature flags.
   * @param {object} flags
   */
  merge(flags) {
    Object.keys(flags).forEach((key) => {
      this.flags.set(key, flags[key]);
    });
  }

  /**
   * @param {FeatureFlagScope} scope
   */
  mergeWithScope(scope) {
    for (const [key, value] of scope.flags) {
      if (this.flags.has(key)) {
        continue;
      }
      this.flags.set(key, value);
    }
  }

  /**
   * Check if a feature flag is enabled
   * @param {string} name
   * @returns {boolean}
   */
  enabled(name) {
    this.checkForFlag(name);
    return this.flags.get(name);
  }
}

const FeatureFlags = createScope();

for (let i = 0; i < featureFlagInfo.length; i++) {
  const featureFlag = featureFlagInfo[i];
  FeatureFlags.add(featureFlag.name, featureFlag.enabled);
}

function createScope(flags) {
  return new FeatureFlagScope(flags);
}

function add(...args) {
  return FeatureFlags.add(...args);
}

function enable(...args) {
  return FeatureFlags.enable(...args);
}

function disable(...args) {
  return FeatureFlags.disable(...args);
}

function enabled(...args) {
  return FeatureFlags.enabled(...args);
}

function merge(...args) {
  return FeatureFlags.merge(...args);
}

export { FeatureFlags, add, createScope, disable, enable, enabled, merge };
