/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import { render, screen } from '@testing-library/react';
import React from 'react';
import { shallow, mount } from 'enzyme';
import RadioButtonGroup from '../RadioButtonGroup';
import RadioButton from '../../RadioButton/next/RadioButton';
describe('RadioButtonGroup', function () {
  it('should render `legendText` in a <label>', function () {
    render( /*#__PURE__*/React.createElement(RadioButtonGroup, {
      defaultSelected: "test-1",
      name: "test",
      legendText: "test"
    }, /*#__PURE__*/React.createElement(RadioButton, {
      labelText: "test-1",
      value: "test-1"
    }), /*#__PURE__*/React.createElement(RadioButton, {
      labelText: "test-2",
      value: "test-2"
    })));
    var legend = screen.getByText('test', {
      selector: 'legend'
    });
    expect(legend).toBeDefined();
  });
  it('should render `legendText` in a <fieldset>', function () {
    render( /*#__PURE__*/React.createElement(RadioButtonGroup, {
      defaultSelected: "test-1",
      name: "test",
      legendText: "test"
    }, /*#__PURE__*/React.createElement(RadioButton, {
      labelText: "test-1",
      value: "test-1"
    }), /*#__PURE__*/React.createElement(RadioButton, {
      labelText: "test-2",
      value: "test-2"
    })));
    var fieldset = screen.getByText('test', {
      selector: 'legend'
    }).closest('fieldset');
    expect(fieldset).toBeDefined();
  });
  it('should render <RadioButton> as children', function () {
    render( /*#__PURE__*/React.createElement(RadioButtonGroup, {
      defaultSelected: "test-1",
      name: "test",
      legendText: "test"
    }, /*#__PURE__*/React.createElement(RadioButton, {
      labelText: "test-1",
      value: "test-1"
    }), /*#__PURE__*/React.createElement(RadioButton, {
      labelText: "test-2",
      value: "test-2"
    })));
    var fieldset = screen.getByText('test', {
      selector: 'legend'
    }).closest('fieldset');
    expect(fieldset).toContainElement(screen.getByLabelText('test-1'));
    expect(fieldset).toContainElement(screen.getByLabelText('test-2'));
  });
  describe('Component API', function () {
    it('should support a custom className on the <fieldset>', function () {
      render( /*#__PURE__*/React.createElement(RadioButtonGroup, {
        className: "custom-class",
        defaultSelected: "test-1",
        name: "test",
        legendText: "test"
      }, /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "test-1",
        value: "test-1"
      }), /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "test-2",
        value: "test-2"
      })));
      var fieldset = screen.getByText('test', {
        selector: 'legend'
      }).closest('fieldset');
      expect(fieldset).toHaveClass('custom-class');
    });
    it('should support passing in disabled to disable the <fieldset>', function () {
      render( /*#__PURE__*/React.createElement(RadioButtonGroup, {
        defaultSelected: "test-1",
        disabled: true,
        name: "test",
        legendText: "test"
      }, /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "test-1",
        value: "test-1"
      }), /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "test-2",
        value: "test-2"
      })));
      var fieldset = screen.getByText('test', {
        selector: 'legend'
      }).closest('fieldset');
      expect(fieldset).toBeDisabled();
    });
    it('should support `defaultSelected` as a way to select a radio button', function () {
      render( /*#__PURE__*/React.createElement(RadioButtonGroup, {
        defaultSelected: "test-1",
        name: "test",
        legendText: "test"
      }, /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "test-1",
        value: "test-1"
      }), /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "test-2",
        value: "test-2"
      })));
      expect(screen.getByLabelText('test-1')).toEqual(screen.getByRole('radio', {
        checked: true
      }));
    });
    it('should support `valueSelected` as a way to select a radio button', function () {
      var _render = render( /*#__PURE__*/React.createElement(RadioButtonGroup, {
        valueSelected: "test-1",
        name: "test",
        legendText: "test"
      }, /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "test-1",
        value: "test-1"
      }), /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "test-2",
        value: "test-2"
      }))),
          rerender = _render.rerender;

      expect(screen.getByLabelText('test-1')).toEqual(screen.getByRole('radio', {
        checked: true
      }));
      rerender( /*#__PURE__*/React.createElement(RadioButtonGroup, {
        valueSelected: "test-2",
        name: "test",
        legendText: "test"
      }, /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "test-1",
        value: "test-1"
      }), /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "test-2",
        value: "test-2"
      })));
      expect(screen.getByLabelText('test-2')).toEqual(screen.getByRole('radio', {
        checked: true
      }));
    });
    it('should support a 0 value for `valueSelected` (#9041)', function () {
      render( /*#__PURE__*/React.createElement(RadioButtonGroup, {
        valueSelected: 0,
        name: "test",
        legendText: "test"
      }, /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "test-1",
        value: 1
      }), /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "test-0",
        value: 0
      })));
      expect(screen.getByLabelText('test-0')).toEqual(screen.getByRole('radio', {
        checked: true
      }));
    });
    describe('onChange event', function () {
      var onChange = jest.fn();
      var wrapper = mount( /*#__PURE__*/React.createElement(RadioButtonGroup, {
        onChange: onChange,
        name: "gender"
      }, /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "Male",
        value: "male"
      }), /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "Female",
        value: "female"
      })));
      var firstRadio = wrapper.find(RadioButton).first();
      var args = ['male', 'gender', {
        test: 'test event'
      }];
      it('first child should not have checked set initially', function () {
        expect(firstRadio.props().checked).toEqual(false);
      });
      it('invoking onChange sets checked on correct child', function () {
        var _firstRadio$props;

        (_firstRadio$props = firstRadio.props()).onChange.apply(_firstRadio$props, args);

        wrapper.update();
        expect(wrapper.find(RadioButton).first().props().checked).toEqual(true);
      });
      it('should invoke onChange with correct arguments', function () {
        var _expect;

        (_expect = expect(onChange)).toHaveBeenCalledWith.apply(_expect, args);
      });
      it('calling onChange with same args should not call onChange prop', function () {
        var _firstRadio$props2;

        onChange.mockClear();

        (_firstRadio$props2 = firstRadio.props()).onChange.apply(_firstRadio$props2, args);

        expect(onChange).not.toHaveBeenCalled();
      });
    });
    describe('Getting derived state from props', function () {
      var wrapper = shallow( /*#__PURE__*/React.createElement(RadioButtonGroup, {
        valueSelected: "male",
        defaultSelected: "female",
        name: "gender"
      }, /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "Male",
        value: "male"
      }), /*#__PURE__*/React.createElement(RadioButton, {
        labelText: "Female",
        value: "female"
      })));
      it('should initialize the current selection from props', function () {
        expect(wrapper.state().selected).toEqual('male');
      });
      it('should change the current selection upon change in props', function () {
        wrapper.setProps({
          valueSelected: 'male'
        });
        wrapper.setState({
          selected: 'male'
        });
        wrapper.setProps({
          valueSelected: undefined
        });
        expect(wrapper.state().selected).toEqual('female');
      });
      it('should avoid change the current selection upon setting props, unless there the value actually changes', function () {
        wrapper.setProps({
          valueSelected: 'female'
        });
        wrapper.setState({
          selected: 'male'
        });
        wrapper.setProps({
          valueSelected: 'female'
        });
        expect(wrapper.state().selected).toEqual('male');
      });
    });
  });
});