/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
import React from 'react';
import Slider from '../Slider';
import SliderSkeleton from '../Slider/Slider.Skeleton';
import { mount, shallow } from 'enzyme';
import 'requestanimationframe';
import throttle from 'lodash.throttle';
import { settings } from 'carbon-components';
jest.mock('lodash.throttle');
throttle.mockImplementation(function (fn) {
  return Object.assign(fn, {
    throttled: true
  });
});
var prefix = settings.prefix;
describe('Slider', function () {
  var id = 'slider';
  var wrapper;
  beforeEach(function () {
    wrapper = mount( /*#__PURE__*/React.createElement(Slider, {
      id: id,
      className: "extra-class",
      value: 1,
      min: 1,
      max: 3,
      step: 1
    }));
  });
  describe('Renders as expected', function () {
    it('renders children as expected', function () {
      expect(wrapper.find(".".concat(prefix, "--text-input")).length).toBe(1);
    });
    it('has the expected classes', function () {
      expect(wrapper.find(".".concat(prefix, "--slider")).length).toBe(1);
    });
    it('renders extra classes passed in via className', function () {
      expect(wrapper.find(".".concat(prefix, "--slider")).hasClass('extra-class')).toEqual(true);
    });
    it('can be disabled', function () {
      wrapper.setProps({
        disabled: true
      });
      expect(wrapper.props().disabled).toEqual(true);
    });
    it('can set value via props', function () {
      wrapper.setProps({
        value: 55
      });
      expect(wrapper.props().value).toEqual(55);
    });
    it('should change the value upon change in props', function () {
      wrapper.setProps({
        value: 1
      });
      wrapper.setState({
        value: 1
      });
      wrapper.update();
      wrapper.setProps({
        value: 2
      });
      expect(wrapper.state().value).toEqual(2);
    });
    it('should accurately position slider on mount', function () {
      expect(wrapper.state().left).toEqual(0);
    });
    it('should specify light version as expected', function () {
      expect(wrapper.props().light).toEqual(false);
      wrapper.setProps({
        light: true
      });
      expect(wrapper.props().light).toEqual(true);
    });
    it('marks input field as hidden if hidden via props', function () {
      wrapper.setProps({
        hideTextInput: true
      });
      expect(wrapper.find("#".concat(id, "-input-for-slider")).props().type).toEqual('hidden');
    });
    it('sets style to display:none on input field if hidden via props', function () {
      wrapper.setProps({
        hideTextInput: true
      });
      expect(wrapper.find("#".concat(id, "-input-for-slider")).props().style).toEqual({
        display: 'none'
      });
    });
  });
  describe('Supporting label', function () {
    it('concatenates the value and the label by default', function () {
      var wrapper = mount( /*#__PURE__*/React.createElement(Slider, {
        min: 0,
        minLabel: "min",
        max: 100,
        maxLabel: "max",
        value: 0
      }));
      expect(wrapper.find(".".concat(prefix, "--slider__range-label")).first().text()).toBe('0min');
      expect(wrapper.find(".".concat(prefix, "--slider__range-label")).last().text()).toBe('100max');
    });
    it('supports custom formatting of the label', function () {
      var wrapper = mount( /*#__PURE__*/React.createElement(Slider, {
        min: 0,
        minLabel: "min",
        max: 100,
        maxLabel: "max",
        formatLabel: function formatLabel(value, label) {
          return "".concat(value, "-").concat(label);
        },
        value: 0
      }));
      expect(wrapper.find(".".concat(prefix, "--slider__range-label")).first().text()).toBe('0-min');
      expect(wrapper.find(".".concat(prefix, "--slider__range-label")).last().text()).toBe('100-max');
    });
  });
  describe('key/mouse event processing', function () {
    var handleChange = jest.fn();
    var handleRelease = jest.fn();
    var handleBlur = jest.fn();
    var wrapper = mount( /*#__PURE__*/React.createElement(Slider, {
      id: "slider",
      className: "extra-class",
      value: 50,
      min: 0,
      max: 100,
      step: 1,
      stepMultiplier: 5,
      onChange: handleChange,
      onRelease: handleRelease,
      onBlur: handleBlur
    }));
    it('sets correct state from event with a right/up keydown', function () {
      var evt = {
        type: 'keydown',
        which: '38'
      };
      wrapper.instance().onKeyDown(evt);
      expect(wrapper.state().value).toEqual(51);
      expect(handleChange).toHaveBeenLastCalledWith({
        value: 51
      });
    });
    it('sets correct state from event with a left/down keydown', function () {
      var evt = {
        type: 'keydown',
        which: '40'
      };
      wrapper.instance().onKeyDown(evt);
      expect(wrapper.state().value).toEqual(50);
      expect(handleChange).toHaveBeenLastCalledWith({
        value: 50
      });
    });
    it('correctly uses setMultiplier with a right/up keydown', function () {
      var evt = {
        type: 'keydown',
        which: '38',
        shiftKey: true
      };
      wrapper.instance().onKeyDown(evt);
      expect(wrapper.state().value).toEqual(55);
      expect(handleChange).toHaveBeenLastCalledWith({
        value: 55
      });
    });
    it('sets correct state from event with a clientX in a mousemove', function () {
      var evt = {
        type: 'mousemove',
        clientX: '1000'
      };

      wrapper.instance()._onDrag(evt);

      expect(handleChange).toHaveBeenLastCalledWith({
        value: 100
      });
      expect(wrapper.state().value).toEqual(100);
    });
    it('sets correct state from event with a clientX in a touchmove', function () {
      var evt = {
        type: 'touchmove',
        touches: [{
          clientX: '0'
        }]
      };

      wrapper.instance()._onDrag(evt);

      expect(handleChange).toHaveBeenLastCalledWith({
        value: 0
      });
      expect(wrapper.state().value).toEqual(0);
    });
    it('throttles mousemove events', function () {
      expect(wrapper.instance().onDrag.throttled).toBe(true);
    });
    describe('user is holding the handle', function () {
      it('does not call onRelease', function () {
        var evt = {
          type: 'mousemove',
          clientX: '1000'
        };
        handleRelease.mockClear();
        expect(handleRelease).not.toHaveBeenCalled();
        wrapper.instance().onDragStart(evt);
        wrapper.instance().onDrag(evt);
        expect(handleRelease).not.toHaveBeenCalled();
      });
    });
    describe('user releases the handle', function () {
      it('calls onRelease', function () {
        handleRelease.mockClear();
        expect(handleRelease).not.toHaveBeenCalled();
        wrapper.setState({
          needsOnRelease: true
        });
        wrapper.instance().onDragStop();
        expect(handleRelease).toHaveBeenCalled();
      });
    });
    it('allows user to set invalid value when typing in input field', function () {
      var evt = {
        target: {
          value: '999'
        }
      };
      wrapper.instance().onChange(evt);
      expect(wrapper.state().value).toEqual(999);
      expect(handleChange).toHaveBeenLastCalledWith({
        value: 999
      });
    });
    it('checks for validity onBlur', function () {
      var checkValidity = jest.fn();
      var evt = {
        target: {
          value: '',
          checkValidity: checkValidity
        }
      };
      wrapper.instance().onBlur(evt);
      expect(checkValidity).toHaveBeenCalled();
    });
    it('sets correct state when typing a valid value in input field', function () {
      var evt = {
        target: {
          value: '12'
        }
      };
      wrapper.instance().onChange(evt);
      expect(wrapper.state().value).toEqual(12);
      expect(handleChange).toHaveBeenLastCalledWith({
        value: 12
      });
    });
  });
  describe('syncs invalid state and props', function () {
    var handleChange = jest.fn();
    var handleRelease = jest.fn();
    var handleBlur = jest.fn();
    var wrapper = mount( /*#__PURE__*/React.createElement(Slider, {
      id: "slider",
      className: "extra-class",
      value: 50,
      min: 0,
      max: 100,
      step: 1,
      onChange: handleChange,
      onRelease: handleRelease,
      onBlur: handleBlur,
      invalid: false
    }));
    it('overrides invalid state for invalid prop', function () {
      var changeEvt = {
        target: {
          value: '200'
        }
      };
      var blurEvt = {
        target: {
          checkValidity: function checkValidity() {
            return false;
          }
        }
      };
      wrapper.instance().onChange(changeEvt);
      wrapper.instance().onBlur(blurEvt);
      expect(wrapper.state().value).toEqual(200);
      expect(wrapper.state().isValid).toEqual(true);
      expect(wrapper.find(".".concat(prefix, "--slider-text-input")).hasClass("".concat(prefix, "--text-input--invalid"))).toEqual(false);
    });
  });
  describe('error handling', function () {
    var handleChange = jest.fn();
    var handleRelease = jest.fn();
    var wrapper = mount( /*#__PURE__*/React.createElement(Slider, {
      id: "slider",
      className: "extra-class",
      value: 50,
      min: 0,
      max: 100,
      step: 1,
      onChange: handleChange,
      onRelease: handleRelease
    }));
    it('handles non-number typed into input field', function () {
      var evt = {
        target: {
          value: ''
        }
      };
      wrapper.instance().onChange(evt);
      expect(wrapper.state().value).toEqual('');
      expect(handleChange).not.toHaveBeenCalled();
    });
    it('gracefully tolerates empty event passed to _onDrag', function () {
      var evt = {};

      wrapper.instance()._onDrag(evt);

      expect(wrapper.state().value).toEqual(''); // from last test

      expect(handleChange).not.toHaveBeenCalled();
    });
    it('gracefully tolerates empty event passed to onChange', function () {
      var evt = {};
      wrapper.instance().onChange(evt);
      expect(wrapper.state().value).toEqual(''); // from last test

      expect(handleChange).not.toHaveBeenCalled();
    });
    it('gracefully tolerates empty event passed to onBlur', function () {
      var evt = {};
      wrapper.instance().onBlur(evt);
      expect(wrapper.state().value).toEqual(''); // from last test

      expect(handleChange).not.toHaveBeenCalled();
    });
    it('gracefully tolerates empty event passed to onKeyDown', function () {
      var evt = {};
      wrapper.instance().onKeyDown(evt);
      expect(wrapper.state().value).toEqual(''); // from last test

      expect(handleChange).not.toHaveBeenCalled();
    });
    it('gracefully tolerates bad key code passed to onKeyDown', function () {
      var evt = {
        which: '123'
      };
      wrapper.instance().onKeyDown(evt);
      expect(wrapper.state().value).toEqual(''); // from last test

      expect(handleChange).not.toHaveBeenCalled();
    });
  });
  describe('slider is disabled', function () {
    var handleChange = jest.fn();
    var handleRelease = jest.fn();
    var wrapper = mount( /*#__PURE__*/React.createElement(Slider, {
      id: "slider",
      className: "extra-class",
      value: 50,
      min: 0,
      max: 100,
      step: 1,
      onChange: handleChange,
      onRelease: handleRelease,
      disabled: true
    }));
    it('does nothing when trying to type in the input', function () {
      var evt = {
        target: {
          value: ''
        }
      };
      wrapper.instance().onChange(evt);
      expect(wrapper.state().value).toEqual(50);
      expect(handleChange).not.toHaveBeenCalled();
    });
    it('does nothing when trying to start a drag', function () {
      var evt = {
        type: 'mousedown',
        clientX: '1001'
      };
      wrapper.instance().onDragStart(evt);
      expect(wrapper.state().value).toEqual(50);
      expect(handleChange).not.toHaveBeenCalled();
    });
    it('does nothing when trying to drag', function () {
      var evt = {
        type: 'mousemove',
        clientX: '1000'
      };

      wrapper.instance()._onDrag(evt);

      expect(wrapper.state().value).toEqual(50);
      expect(handleChange).not.toHaveBeenCalled();
    });
    it('does nothing when trying to stop a drag', function () {
      var evt = {
        type: 'mouseup',
        clientX: '1001'
      };
      wrapper.instance().onDragStop(evt);
      expect(wrapper.state().needsOnRelease).toEqual(false);
      expect(handleChange).not.toHaveBeenCalled();
      expect(handleRelease).not.toHaveBeenCalled();
    });
    it('does nothing when using arrow key', function () {
      var evt = {
        type: 'keydown',
        which: '40'
      };
      wrapper.instance().onKeyDown(evt);
      expect(wrapper.state().value).toEqual(50);
      expect(handleChange).not.toHaveBeenCalled();
    });
  });
});
describe('SliderSkeleton', function () {
  describe('Renders as expected', function () {
    var wrapper = shallow( /*#__PURE__*/React.createElement(SliderSkeleton, null));
    var slider = wrapper.find(".".concat(prefix, "--slider-container"));
    it('Has the expected classes', function () {
      expect(slider.hasClass("".concat(prefix, "--skeleton"))).toEqual(true);
    });
  });
});