/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
// FIXME:
/* eslint-disable react/prop-types */
import Debug from 'debug';
import React from 'react';
import { i18n, pexecInCurrentTab, encodeComponent, inBrowser, eventChannelUnsafe, findThemeByName, getPersistedThemeChoice, getDefaultTheme } from '@kui-shell/core';
import Card from '../spi/Card';
import KuiContext from './context';
import StatusStripe from './StatusStripe';
import { TabContainer, Loading, Alert } from '../..';
import loadUserSettings, { on as onUserSettingsChange } from './UserSettings';
const InputStripe = React.lazy(() => import('./InputStripe'));
import KuiIcon from '../../../icons/png/WelcomeLight.png';
const debug = Debug('<Kui/>');
const strings = i18n('client');
const Popup = React.lazy(() => import(/* webpackMode: "lazy" */ './Popup'));
const defaultThemeProperties = {
    components: 'patternfly',
    topTabNames: 'fixed'
};
/**
 * Render the main body of our client.
 *
 * |Notes on Session Initialization|: to provide custom views for
 * session initialization (only relevant for browser-based hosted
 * Kui), you can instantiate <Kui/> with these properties (defined in
 * KuiConfiguration), show here with some sample views:
 *
 * <Kui
 *    loading={<div className="kui--hero-text">Hold on...</div>}
 *    reinit={<div className="kui--hero-text">Connection broken...</div>}
 *    loadingError={err => <div className="kui--hero-text">{err.toString()}</div>}
 *    loadingDone={<div>Welcome to Kui</div>}
 * />
 *
 */
export class Kui extends React.PureComponent {
    constructor(props) {
        super(props);
        this.firstTab = true;
        this._onTabReady = this.onTabReady.bind(this);
        // refresh the UI after user changes settings overrides
        onUserSettingsChange(evt => {
            this.setState(curState => {
                // a bit of care is needed to handle removing keys
                if (evt !== 'set') {
                    for (const key in curState.userOverrides) {
                        delete curState[key];
                    }
                }
                const newUserOverrides = loadUserSettings();
                return Object.assign(newUserOverrides, { userOverrides: newUserOverrides });
            });
        });
        eventChannelUnsafe.on('/theme/change', this.onThemeChange.bind(this));
        setTimeout(() => __awaiter(this, void 0, void 0, function* () {
            const { theme } = yield findThemeByName((yield getPersistedThemeChoice()) || (yield getDefaultTheme()));
            this.setState(curState => {
                const stateWithThemeProps = Object.assign({}, theme, curState);
                debug('state with theme props', stateWithThemeProps);
                return stateWithThemeProps;
            });
        }));
        if (!props.noBootstrap) {
            import('@kui-shell/core')
                .then(_ => _.bootIntoSandbox())
                .then(() => {
                this.setState({ isBootstrapped: true });
            });
        }
        let commandLine = this.props.commandLine;
        let quietExecCommand = this.props.quietExecCommand !== undefined ? this.props.quietExecCommand : !this.props.isPopup;
        if (inBrowser()) {
            const windowQuery = window.location.search;
            if (windowQuery) {
                // parse and extract the question mark in window.location.search
                // e.g. query = { command: 'replay /kui/welcome.json' }
                const query = require('querystring').parse(windowQuery.substring(1));
                // To avoid SQL injection attacks, users can only query with `replay` command
                if (query.command && /^replay/.test(query.command)) {
                    commandLine = query.command.split(' ');
                    quietExecCommand = false;
                }
            }
        }
        try {
            const userOverrides = loadUserSettings();
            this.state = Object.assign({}, this.defaultSessionBehavior(), this.defaultFeatureFlag(), props, userOverrides, {
                userOverrides,
                isBootstrapped: !!props.noBootstrap,
                commandLine,
                quietExecCommand
            });
            debug('initial state:inBrowser?', inBrowser());
            debug('initial state:given properties', props);
            debug('initial state:final value', this.state);
        }
        catch (err) {
            console.log('using default configuration');
            this.state = {
                isBootstrapped: !!props.noBootstrap,
                commandLine,
                quietExecCommand
            };
        }
    }
    defaultFeatureFlag() {
        return {
            sidecarName: 'breadcrumb',
            showWelcomeMax: -1
        };
    }
    defaultLoading() {
        return (React.createElement(Loading, { className: "somewhat-larger-text", description: strings('Please wait while we connect to your cluster') }));
    }
    defaultReinit() {
        return (React.createElement(Alert, { hideCloseButton: true, className: "kui--terminal-alert kui--connection-lost", alert: {
                type: 'error',
                title: strings('Lost connection to your cluster'),
                body: strings('Attempting to reconnect...')
            } }));
    }
    defaultLoadingDone() {
        return (repl) => !inBrowser() ? (undefined) : (React.createElement(Card, { titleInHeader: true, bodyInHeader: true, title: strings('Successfully connected to your cluster'), repl: repl, icon: KuiIcon }, strings('loadingDone:content')));
    }
    defaultLoadingError() {
        return err => (React.createElement(Alert, { alert: { type: 'error', title: strings('Error connecting to your cluster'), body: err.toString() } }));
    }
    /**
     * For browser-based clients, this defines the default UI for
     * session initialization.
     *
     */
    defaultSessionBehavior() {
        const behavior = !inBrowser()
            ? {}
            : {
                loading: this.defaultLoading(),
                reinit: this.defaultReinit(),
                loadingError: this.defaultLoadingError(),
                loadingDone: this.defaultLoadingDone()
            };
        debug('defaultSessionBehavior', behavior);
        return behavior;
    }
    onThemeChange({ themeModel }) {
        this.setState(curState => {
            // note the priority order, from highest to lowest:
            //  1) any properties defined by the theme (since we just switched themes)
            //  2) any properties defined by the container of this <Kui/>
            //  3) default choices
            //  4) any prior state
            // re: ordering of 3 and 4, see https://github.com/IBM/kui/issues/4423
            const stateAfterThemeChange = Object.assign({}, curState, defaultThemeProperties, this.props, themeModel);
            debug('state after theme change', stateAfterThemeChange);
            return stateAfterThemeChange;
        });
    }
    /**
     * Props to pass to StatusStripe. This allows us to set the desired
     * status stripe color at startup time, rather than seeing the
     * default color, followed quickly by a change to the color desired
     * by the controller backing the given `props.commandLine`. The
     * controller may still want to specialize the status stripe
     * further, but at least we can avoid that odd
     * e.g. defaultColor-then-blue effect.
     *
     */
    statusStripeProps() {
        if (this.state.commandLine) {
            const statusStripeIdx = this.state.commandLine.findIndex(_ => _ === '--status-stripe');
            if (statusStripeIdx >= 0) {
                return { type: this.props.commandLine[statusStripeIdx + 1] };
            }
        }
    }
    componentDidCatch(error, errorInfo) {
        console.error(error, errorInfo);
    }
    onTabReady() {
        if (this.state.commandLine && this.firstTab) {
            this.firstTab = false;
            pexecInCurrentTab(this.state.commandLine.map(_ => encodeComponent(_)).join(' '), undefined, this.state.quietExecCommand);
        }
    }
    render() {
        if (!this.state.isBootstrapped) {
            return React.createElement(Loading, null);
        }
        if (this.props.isPopup && this.state.commandLine) {
            return (React.createElement(KuiContext.Provider, { value: this.state },
                React.createElement(React.Suspense, { fallback: React.createElement("div", null) },
                    React.createElement(Popup, { commandLine: this.state.commandLine, noHelp: this.props.noHelp, noSettings: this.props.noSettings }, this.props.children))));
        }
        else {
            const bottom = !!this.props.bottomInput && React.createElement(InputStripe, null, this.props.bottomInput);
            return (React.createElement(KuiContext.Provider, { value: this.state },
                React.createElement(React.Suspense, { fallback: React.createElement("div", null) },
                    React.createElement("div", { className: "kui--full-height" },
                        React.createElement(TabContainer, { noActiveInput: this.props.noActiveInput || !!this.props.bottomInput, bottom: bottom, title: this.props.initialTabTitle, onTabReady: this.state.commandLine && this._onTabReady }),
                        this.props.toplevel,
                        this.props.statusStripe !== false && (React.createElement(StatusStripe, Object.assign({ noHelp: this.props.noHelp, noSettings: this.props.noSettings }, this.statusStripeProps()), this.props.children))))));
        }
    }
}
export default Kui;
//# sourceMappingURL=Kui.js.map