"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.assertInSandbox = assertInSandbox;
exports.inSandbox = inSandbox;
exports.assertLocalAccess = exports.hasLocalAccess = exports.hasAuth = exports.setValidCredentials = exports.getValidCredentials = exports.getAuthValue = exports.getAuth = exports.setHasAuth = exports.inProxy = exports.assertHasProxy = exports.hasProxy = exports.inBrowser = exports.inElectron = exports.isHeadless = exports.getMedia = exports.setMedia = exports.Media = void 0;

var _debug = _interopRequireDefault(require("debug"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2018 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* eslint-disable @typescript-eslint/explicit-member-accessibility */
const debug = (0, _debug.default)('core/capabilities');
debug('loading');
/**
 * Are we running headless, in electron, or in a browser?
 *
 */

var Media;
exports.Media = Media;

(function (Media) {
  Media[Media["Unknown"] = 0] = "Unknown";
  Media[Media["Headless"] = 1] = "Headless";
  Media[Media["Electron"] = 2] = "Electron";
  Media[Media["Browser"] = 3] = "Browser";
})(Media || (exports.Media = Media = {}));
/**
 * Current state of capabilities
 *
 */


class State {
  constructor() {
    this.inSandbox = false;
    this.assertedLocalAccess = false;
    this.hasLocalAccess = true; // may change as media changes or assertLocalAccess is called

    this.hasProxy = false;
    this.media = Media.Unknown;
    this.validCredentials = {}; // map to the credentials
  }

}

const state = new State();
/**
 * Update the media, e.g. to indicate that we are running in a browser
 * context versus an Electron context.
 *
 */

const setMedia = media => {
  debug('setMedia %s', Media[media]);
  state.media = media;

  if (!state.assertedLocalAccess && media === Media.Browser) {
    state.hasLocalAccess = false;
  }
};
/**
 * What is our presentation media?
 *
 */


exports.setMedia = setMedia;

const getMedia = () => state.media;

exports.getMedia = getMedia;

const isHeadless = () => state.media === Media.Headless;

exports.isHeadless = isHeadless;

const inElectron = () => state.media === Media.Electron;

exports.inElectron = inElectron;

const inBrowser = () => {
  if (state.media === Media.Browser) {
    return true;
  }

  if (!isHeadless() && typeof document !== 'undefined' && document.body.classList.contains('not-electron')) {
    setMedia(Media.Browser);
    return true;
  } else {
    return false;
  }
};
/**
 * Is Kui supported by a remote proxy?
 *
 */


exports.inBrowser = inBrowser;

const hasProxy = () => state.hasProxy;
/**
 * Assert that Kui is supported by a remote proxy
 *
 */


exports.hasProxy = hasProxy;

const assertHasProxy = () => {
  state.hasProxy = true;
};
/**
 * Are we the Kui proxy?
 *
 */


exports.assertHasProxy = assertHasProxy;

const inProxy = () => {
  return process.env.KUI_REPL_MODE !== undefined;
};
/**
 * Yes, we are running in a sandbox
 *
 */


exports.inProxy = inProxy;

function assertInSandbox() {
  state.inSandbox = true;
}
/**
 * Are we running in a sandbox?
 *
 */


function inSandbox() {
  return state.inSandbox;
}
/**
 * Yes, we have valid credentials to interface with the given
 * provider
 *
 */


const setHasAuth = (provider, creds) => {
  debug('setHasAuth', provider, creds);
  state.validCredentials[provider] = creds;
};
/**
 * Retrieve the auth model for the given provider
 *
 */


exports.setHasAuth = setHasAuth;

const getAuth = provider => state.validCredentials[provider];
/**
 * Retrieve the value for the given key for the auth model of the given provider
 *
 */


exports.getAuth = getAuth;

const getAuthValue = (provider, key) => {
  const model = state.validCredentials[provider];
  return model && model[key];
};
/**
 * Return a map of all valid credentials
 *
 */


exports.getAuthValue = getAuthValue;

const getValidCredentials = () => state.validCredentials;
/**
 * Inject the credentials map
 *
 */


exports.getValidCredentials = getValidCredentials;

const setValidCredentials = creds => {
  debug('setValidCredentials', creds);
  state.validCredentials = creds;
};
/**
 * Do we have valid credentials to interface with the given provider?
 *
 */


exports.setValidCredentials = setValidCredentials;

const hasAuth = provider => !!state.validCredentials[provider];
/**
 * Do we have access to a local system?
 *
 */


exports.hasAuth = hasAuth;

const hasLocalAccess = () => {
  return state.hasLocalAccess;
};
/**
 * Assert that we have local access, even if the default behavior
 * would indicate otherwise
 *
 */


exports.hasLocalAccess = hasLocalAccess;

const assertLocalAccess = () => {
  state.hasLocalAccess = true;
};

exports.assertLocalAccess = assertLocalAccess;