/*
 * Copyright 2017 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* eslint-disable @typescript-eslint/explicit-member-accessibility */
/** generic clone */
function clone(instance) {
    const copy = new instance.constructor();
    Object.assign(copy, instance);
    return copy;
}
class ClassList {
    constructor() {
        this.classList = [];
    }
    get length() {
        return this.classList.length;
    }
    forEach(fn) {
        this.classList.forEach(fn);
    }
    add(_) {
        return this.classList.push(_);
    }
    contains(_) {
        return this.classList.indexOf(_) >= 0;
    }
    remove(_) {
        const idx = this.classList.findIndex((x) => x === _);
        if (idx >= 0) {
            this.classList.splice(idx, 1);
        }
    }
}
class ElementMimic {
    constructor() {
        this._isFakeDom = true;
        this.value = '';
        this.innerText = '';
        this.innerHTML = '';
        this.className = '';
        this.classList = new ClassList();
        this.nodeType = '';
        this.style = {};
        this.children = [];
        this.cells = [];
        this.rows = [];
        this._attrs = {};
    }
    focus() {
        /* empty ok */
    }
    appendChild(c) {
        return this.children.push(c);
    }
    getAttribute(k) {
        return this._attrs[k] || '';
    }
    setAttribute(k, v) {
        this._attrs[k] = v;
        return v;
    }
    remove() {
        // no-op for now
    }
    removeAttribute(k) {
        const attr = this._attrs[k];
        delete this._attrs[k];
        return attr;
    }
    cloneNode() {
        return clone(this);
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    querySelectorAll(selector) {
        return [];
    }
    querySelector() {
        return new ElementMimic();
    }
    addEventListener() {
        return true;
    }
    hasStyle(style, desiredValue) {
        const actualValue = this.style && this.style[style];
        // intentional double equals, so that 500=='500'
        // eslint-disable-next-line eqeqeq
        if (desiredValue)
            return desiredValue == actualValue;
        else
            return actualValue;
    }
    recursiveInnerTextLength() {
        return (this.innerText.length + this.children.reduce((sum, child) => sum + child.recursiveInnerTextLength(), 0));
    }
    insertRow(idx) {
        const row = new ElementMimic();
        row.nodeType = 'tr';
        row.cells = [];
        if (idx === -1)
            this.rows.push(row);
        else
            this.rows.splice(idx, 0, row);
        return row;
    }
    insertCell(idx) {
        const cell = new ElementMimic();
        cell.nodeType = 'td';
        if (idx === -1)
            this.cells.push(cell);
        else
            this.cells.splice(idx, 0, cell);
        return cell;
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    static isFakeDom(dom) {
        return dom && dom._isFakeDom;
    }
}
export default ElementMimic;
//# sourceMappingURL=element-mimic.js.map