"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = makeWatchable;
exports.SingleKindDirectWatcher = exports.DirectWatcher = void 0;

var _debug = _interopRequireDefault(require("debug"));

var _group = require("./group");

var _columns = _interopRequireDefault(require("./columns"));

var _headers = require("./headers");

var _states = require("../../../lib/model/states");

var _formatTable = require("../../../lib/view/formatTable");

var _fetchFile = require("../../../lib/util/fetch-file");

var _resource = require("../../../lib/model/resource");

var _status = require("../../kubectl/status");

var _config = require("../../kubectl/config");

var _options = require("../../kubectl/options");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};

const debug = (0, _debug.default)('plugin-kubectl/client/direct/watch');

class DirectWatcher {
  constructor() {
    /** The current stream jobs. These will be aborted/flow-controlled as directed by the associated view. */
    this.jobs = [];
  }
  /** This will be called by the view when it wants the underlying streamer to resume flowing updates */


  xon() {
    this.jobs.forEach(job => job.xon());
  }
  /** This will be called by the view when it wants the underlying streamer to stop flowing updates */


  xoff() {
    this.jobs.forEach(job => job.xoff());
  }
  /** This will be called by the view when it wants the underlying streamer to die */


  abort() {
    debug('abort requested', this.jobs.length, this.jobs);
    this.jobs.forEach(job => job.abort());
  }
  /** This will be called by the streamer when the underlying job has exited */


  onExit(exitCode) {
    debug('job exited', exitCode); // for now, we don't have much to do here
  }
  /** This will be called by the streamer when it is ready to start flowing data. */


  onReady() {
    // for now, we don't have much to do here
    debug('job is ready');
  }

}
/**
 * This class provides an implementation of a Watcher extension of a
 * Kui Table. It establishes a stream to the apiServer, and hooks the emitted stream of WatchUpdate events into the WatchPusher API of the table: i.e. the this.pusher.offline(), update(), header(), and footer() calls.
 *
 */


exports.DirectWatcher = DirectWatcher;

class SingleKindDirectWatcher extends DirectWatcher {
  constructor(drilldownCommand, args, kind, resourceVersion, formatUrl, group, finalState, initialRowKeys, nNotReady, // number of resources to wait on
  monitorEvents = true, needsStatusColumn = false) {
    super();
    this.drilldownCommand = drilldownCommand;
    this.args = args;
    this.kind = kind;
    this.resourceVersion = resourceVersion;
    this.formatUrl = formatUrl;
    this.group = group;
    this.finalState = finalState;
    this.nNotReady = nNotReady;
    this.monitorEvents = monitorEvents;
    this.needsStatusColumn = needsStatusColumn;
    this.alreadySentHeaders = false;

    if (finalState) {
      // populate the readyDebouncer using the initial set of rows
      this.readyDebouncer = initialRowKeys ? initialRowKeys.reduce((M, {
        rowKey,
        isReady
      }) => {
        M[rowKey] = isReady;
        return M;
      }, {}) : undefined;
    }
  }
  /** This will be called by the view when it is ready to accept push updates */


  init(pusher) {
    return __awaiter(this, void 0, void 0, function* () {
      this.pusher = pusher;
      yield Promise.all([this.initBodyUpdates(), this.monitorEvents && this.kind !== 'Event' ? this.initFooterUpdates() : Promise.resolve()]);
    });
  }

  isCustomColumns() {
    return (0, _options.isCustomColumns)((0, _options.formatOf)(this.args));
  }
  /** Initialize the streamer for main body updates; i.e. for the rows of the table */


  initBodyUpdates() {
    const url = this.formatUrl(true) + `?watch=true&resourceVersion=${this.resourceVersion.toString()}`;
    const onInit = this.onInitForBodyUpdates.bind(this);
    return (0, _fetchFile.openStream)(this.args, url, this.mgmt(onInit), this.isCustomColumns() ? _headers.headersForPlainRequest : _headers.headersForTableRequest);
  }

  formatEventUrl(watchOpts) {
    const fieldSelector = `fieldSelector=involvedObject.kind=${this.kind}`;
    return this.formatUrl(true, undefined, undefined, {
      version: 'v1',
      kind: 'Event'
    }) + `?${fieldSelector}` + (!watchOpts ? '' : `&watch=true&resourceVersion=${watchOpts.resourceVersion.toString()}`);
  }

  filterFooterRows(table, nameColumnIdx, objectColumnIdx) {
    const rows = table.rows.filter(row => {
      const kindAndName = row.cells[objectColumnIdx];

      if (typeof kindAndName === 'string') {
        const kind = kindAndName.split('/')[0];
        const name = kindAndName.split('/')[1];
        return (0, _group.isObjectInGroup)(this.group, kind, name);
      }
    });
    const last = this.lastFooterEvents;
    this.lastFooterEvents = rows.reduce((M, row) => {
      M[row.cells[nameColumnIdx]] = true;
      return M;
    }, {});

    if (!last) {
      return rows;
    } else {
      return rows.filter(row => !last[row.cells[nameColumnIdx]]);
    }
  }
  /** Format a MetaTable of events into a string[] */


  formatFooter(table) {
    if (!this.footerColumnDefinitions) {
      console.error('Dropping footer update, due to missing column definitions');
    } else {
      const {
        lastSeenColumnIdx,
        objectColumnIdx,
        messageColumnIdx,
        nameColumnIdx
      } = this.footerColumnDefinitions;
      return this.filterFooterRows(table, nameColumnIdx, objectColumnIdx).map(_ => {
        const lastSeen = _.cells[lastSeenColumnIdx];
        const involvedObjectName = _.cells[objectColumnIdx];
        const message = _.cells[messageColumnIdx];
        const eventName = _.cells[nameColumnIdx];
        const onClick = `#kuiexec?command=${encodeURIComponent((0, _options.withKubeconfigFrom)(this.args, `kubectl get event ${eventName} -o yaml`))}`;
        return `[[${lastSeen}]](${onClick})` + ` **${involvedObjectName}**: ${message}`;
      });
    }
  }
  /** We pre-process the columnDefinitions for the events, to pick out the column indices of interest. */


  initFooterColumnDefinitions(columnDefinitions) {
    const indices = columnDefinitions.reduce((indices, _, idx) => {
      if (_.name === 'Last Seen') {
        indices.lastSeenColumnIdx = idx;
      } else if (_.name === 'Object') {
        indices.objectColumnIdx = idx;
      } else if (_.name === 'Message') {
        indices.messageColumnIdx = idx;
      } else if (_.name === 'Name') {
        indices.nameColumnIdx = idx;
      }

      return indices;
    }, {
      lastSeenColumnIdx: -1,
      objectColumnIdx: -1,
      messageColumnIdx: -1,
      nameColumnIdx: -1
    });
    const {
      lastSeenColumnIdx,
      objectColumnIdx,
      messageColumnIdx,
      nameColumnIdx
    } = indices;

    if (lastSeenColumnIdx < 0) {
      console.error('Unable to process footer column definitions, due to missing Last Seen column', columnDefinitions);
    } else if (objectColumnIdx < 0) {
      console.error('Unable to process footer column definitions, due to missing Object column', columnDefinitions);
    } else if (messageColumnIdx < 0) {
      console.error('Unable to process footer column definitions, due to missing Message column', columnDefinitions);
    } else if (nameColumnIdx < 0) {
      console.error('Unable to process footer column definitions, due to missing Name column', columnDefinitions);
    } else {
      this.footerColumnDefinitions = indices;
    }
  }
  /** This will be called by the event streamer when it has new data */


  onEventData(update) {
    if (!this.footerColumnDefinitions && update.object.columnDefinitions) {
      this.initFooterColumnDefinitions(update.object.columnDefinitions);
    }

    this.pusher.footer(this.formatFooter(update.object));
  }
  /** Initialize the streamer for table footer updates */


  initFooterUpdates() {
    return __awaiter(this, void 0, void 0, function* () {
      // first: we need to fetch the initial table (so that we have a resourceVersion)
      const events = (yield (0, _fetchFile.fetchFile)(this.args.REPL, this.formatEventUrl(), {
        headers: _headers.headersForTableRequest
      }))[0];

      if ((0, _resource.isMetaTable)(events)) {
        this.onEventData({
          object: events
        }); // second: now we can start the streamer against that resourceVersion

        const watchUrl = this.formatEventUrl({
          resourceVersion: events.metadata.resourceVersion
        });
        return (0, _fetchFile.openStream)(this.args, watchUrl, {
          onInit: job => {
            this.jobs.push(job);
            return this.onEventData.bind(this);
          }
        }, _headers.headersForTableRequest);
      } else {
        console.error('Unexpected response from event query', this.formatEventUrl(), events);
      }
    });
  }
  /** This is the stream management bits for the body */


  mgmt(onInit) {
    return {
      onInit,
      onReady: this.onReady.bind(this),
      onExit: this.onExit.bind(this)
    };
  }
  /** The streamer is almost ready. We give it back a stream to push data to */


  onInitForBodyUpdates(job) {
    this.jobs.push(job);
    return this.onData.bind(this);
  }
  /** This will be called whenever the streamer has data for us. */


  onData(update) {
    return __awaiter(this, void 0, void 0, function* () {
      if (!update.object.columnDefinitions && this.bodyColumnDefinitions) {
        update.object.columnDefinitions = this.bodyColumnDefinitions;
      }

      if (!update.object.columnDefinitions && !this.isCustomColumns()) {
        console.error('Missing column definitions in update', update);
        return;
      }

      const custo = (0, _options.isCustomColumns)((0, _options.formatOf)(this.args));
      let sendHeaders = !this.alreadySentHeaders;

      if (!custo) {
        if (!this.bodyColumnDefinitions) {
          sendHeaders = true;
          this.bodyColumnDefinitions = update.object.columnDefinitions;
        }

        update.object.rows = update.object.rows.filter(row => (0, _group.isObjectInGroup)(this.group, this.kind, row.object.metadata.name));
      }

      const table = custo ? (yield Promise.resolve().then(() => require('./custom-columns'))).toKuiTableForUpdateFromCustomColumns(update.object, this.args, this.drilldownCommand, this.kind, (0, _options.formatOf)(this.args)) : (0, _formatTable.toKuiTable)(update.object, this.kind, this.args, this.drilldownCommand, this.needsStatusColumn, (0, _columns.default)(this.kind, this.args));

      if (sendHeaders) {
        this.alreadySentHeaders = true;
        this.pusher.header(table.header);
      }

      table.body.forEach((row, idx) => {
        const rowNeverSeenBefore = this.readyDebouncer && this.readyDebouncer[row.rowKey] === undefined;

        if (rowNeverSeenBefore) {
          debug('dropping untracked row', row.rowKey);
          return;
        }

        if (update.type === 'ADDED' || update.type === 'MODIFIED') {
          this.pusher.update(row, true);
        } else {
          this.pusher.offline(row.rowKey);
        }

        if (this.kind === 'Namespace') {
          if (update.type === 'ADDED') {
            (0, _config.emitKubectlConfigChangeEvent)('CreateOrDeleteNamespace', row.name);
          } else if (update.type === 'DELETED') {
            (0, _config.emitKubectlConfigChangeEvent)('CreateOrDeleteNamespace', row.name);
          }
        }

        this.checkIfReady(row, idx, update);
      });
      this.pusher.batchUpdateDone();
    });
  }
  /**
   * If we were asked to watch for resources reaching a given
   * `this.finalState`, then check the resource represented by the
   * given Row against the desired final state.
   *
   */


  checkIfReady(row, idx, update) {
    if (this.finalState && this.nNotReady > 0) {
      const isReady = // NOPE: could be added and still not ready (update.type === 'ADDED' && this.finalState === FinalState.OnlineLike) ||
      update.type === 'DELETED' && this.finalState === _states.FinalState.OfflineLike || (update.type === 'ADDED' || update.type === 'MODIFIED') && (0, _status.isResourceReady)(row, this.finalState);
      debug('checking if resource is ready', isReady, this.finalState, this.nNotReady, row, update.type, update.object.rows[idx]);

      if (this.readyDebouncer && isReady && !this.readyDebouncer[row.rowKey]) {
        debug('A resource is in its final state', row.name, this.nNotReady);
        this.readyDebouncer[row.rowKey] = true;

        if (--this.nNotReady <= 0) {
          debug('All resources are in their final state');
          this.abort();
          this.pusher.done();
        }
      }
    }
  }

}
/**
 * If possible, turn a table into a Table & Watchable. If the given
 * `table` does not have a `resourceVersion` attribute, this mapping
 * will not be possible.
 *
 */


exports.SingleKindDirectWatcher = SingleKindDirectWatcher;

function makeWatchable(drilldownCommand, args, kind, group, table, formatUrl, finalState, initialRowKeys, nNotReady, monitorEvents = true, needsStatusColumn = false) {
  return __awaiter(this, void 0, void 0, function* () {
    if (!table.resourceVersion) {
      // we need a cursor to start watching
      console.error('Cannot start watching, due to missing resourceVersion');
      return table;
    }

    return Object.assign(table, {
      watch: new SingleKindDirectWatcher(drilldownCommand, args, kind, table.resourceVersion, formatUrl, group, finalState, initialRowKeys, nNotReady, monitorEvents, needsStatusColumn)
    });
  });
}