"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCurrentContext = getCurrentContext;
exports.getAllContexts = getAllContexts;
exports.getCurrentContextName = getCurrentContextName;
exports.getCurrentDefaultNamespace = getCurrentDefaultNamespace;
exports.getCurrentDefaultContextName = getCurrentDefaultContextName;
exports.default = void 0;

var _flags = _interopRequireDefault(require("./flags"));

var _apiVersion = _interopRequireDefault(require("./apiVersion"));

var _exec = require("./exec");

var _help = require("../../lib/util/help");

var _config = require("./config");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2018 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};

const usage = {
  context: command => ({
    command,
    strict: command,
    docs: 'Print your current kubernetes context',
    example: 'kubectl context'
  }),
  contexts: command => ({
    command,
    strict: command,
    docs: 'List your available kubernetes contexts',
    optional: [{
      name: '-o',
      docs: 'Output format',
      allowed: ['wide']
    }],
    example: 'kubectl contexts'
  })
};
/** Extract the cell value for the given column name (`key`) in the given `row` */

function valueOf(key, row) {
  const cell = row.attributes.find(_ => _.key === key);
  return cell ? cell.value : '';
}
/**
 * @return a `KubeContext` representing the current context
 *
 */


function getCurrentContext({
  REPL
}) {
  return __awaiter(this, void 0, void 0, function* () {
    // fetch both the current context name, and the list of KubeContext objects */
    const [currentContextName, {
      content: contexts
    }] = yield Promise.all([REPL.qexec(`context`), REPL.rexec(`contexts`)]); // the KubeContext object matching the current context name

    return contexts.find(_ => _.metadata.name === currentContextName);
  });
}
/** @return a list of `KubeContext` for all known contexts */


function getAllContexts({
  REPL
}) {
  return __awaiter(this, void 0, void 0, function* () {
    return (yield REPL.rexec('contexts')).content;
  });
}

function getCurrentContextName({
  REPL
}) {
  return __awaiter(this, void 0, void 0, function* () {
    const context = yield REPL.qexec('kubectl config current-context');
    return context ? context.trim() : context;
  });
}
/** Extract the namespace from the current context */


let currentDefaultNamespaceCache;
let currentDefaultContextCache;
(0, _config.onKubectlConfigChangeEvents)((type, namespace, context) => {
  if (type === 'SetNamespaceOrContext') {
    if (typeof namespace === 'string' && namespace.length > 0) {
      currentDefaultNamespaceCache = Promise.resolve(namespace);
    } else {
      // invalidate cache
      currentDefaultNamespaceCache = undefined;
    }

    if (typeof context === 'string' && context.length > 0) {
      currentDefaultContextCache = Promise.resolve(context);
    } else {
      // invalidate cache
      currentDefaultContextCache = undefined;
    }
  }
});

function getCurrentDefaultNamespace({
  REPL
}) {
  return __awaiter(this, void 0, void 0, function* () {
    if (currentDefaultNamespaceCache) {
      return currentDefaultNamespaceCache;
    } // eslint-disable-next-line no-async-promise-executor


    currentDefaultNamespaceCache = new Promise(resolve => __awaiter(this, void 0, void 0, function* () {
      const cmdline = `kubectl config view --minify --output "jsonpath={..namespace}"`;
      const ns = yield REPL.qexec(cmdline).then(_ns => {
        const ns = typeof _ns === 'number' ? _ns.toString() : _ns; // ns might be number

        if (typeof ns !== 'string' || ns.length === 0) {
          // e.g. microk8s
          console.error('Suspicious return value for current namespace', ns, cmdline);
          return 'default';
        } else {
          return ns;
        }
      }).catch(err => {
        if (err.code !== 404 && !/command not found/.test(err.message)) {
          console.error('error determining default namespace', err);
        }

        resolve('default');
      });
      resolve(ns ? ns.trim() : 'default');
    }));
    return currentDefaultNamespaceCache;
  });
}

function getCurrentDefaultContextName({
  REPL
}) {
  if (currentDefaultContextCache) {
    return currentDefaultContextCache;
  } // eslint-disable-next-line no-async-promise-executor


  currentDefaultContextCache = new Promise(resolve => __awaiter(this, void 0, void 0, function* () {
    const context = yield getCurrentContextName({
      REPL
    }).catch(err => {
      if (err.code !== 404 && !/command not found/.test(err.message)) {
        console.error('error determining default context', err);
      }

      return '';
    });
    resolve(context);
  }));
  return currentDefaultContextCache;
}
/**
 * List contets command handler
 *
 */


const listContexts = args => __awaiter(void 0, void 0, void 0, function* () {
  const execOptions = Object.assign({}, args.execOptions, {
    render: false
  });
  const contexts = yield args.REPL.qexec(`kubectl config get-contexts`, undefined, undefined, execOptions);

  if (args.execOptions.raw) {
    return {
      mode: 'raw',
      content: contexts.body.map(_ => ({
        apiVersion: _apiVersion.default,
        kind: 'Context',
        originatingCommand: args,
        isKubeResource: true,
        metadata: {
          name: valueOf('NAME', _),
          namespace: valueOf('NAMESPACE', _)
        },
        spec: {
          user: valueOf('AUTHINFO', _),
          cluster: valueOf('CLUSTER', _),
          isCurrent: _.rowCSS === 'selected-row' || Array.isArray(_.rowCSS) && _.rowCSS.indexOf('selected-row') >= 0
        }
      }))
    };
  } else {
    return contexts;
  }
});
/**
 * Register the commands
 *
 */


var _default = commandTree => {
  commandTree.listen('/kubectl/config/get-contexts', args => (0, _help.isUsage)(args) ? (0, _help.doHelp)('kubectl', args) : (0, _exec.doExecWithTable)(args, undefined, undefined, {
    entityType: 'Context'
  }), _flags.default);
  commandTree.listen('/context', ({
    REPL
  }) => __awaiter(void 0, void 0, void 0, function* () {
    return (yield REPL.qexec('kubectl config current-context')).trim();
  }), Object.assign({
    usage: usage.context('context')
  }, _flags.default));
  commandTree.listen('/contexts', listContexts, Object.assign({
    usage: usage.contexts('contexts')
  }, _flags.default));
};

exports.default = _default;