"use strict";

var _react = _interopRequireDefault(require("react"));

var _enzyme = require("enzyme");

var _testHelpers = require("../ListBox/test-helpers");

var _ComboBox = _interopRequireDefault(require("../ComboBox"));

var _carbonComponents = require("carbon-components");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); if (enumerableOnly) { symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; }); } keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; if (i % 2) { ownKeys(Object(source), true).forEach(function (key) { _defineProperty(target, key, source[key]); }); } else if (Object.getOwnPropertyDescriptors) { Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)); } else { ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function _extends() { _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; return _extends.apply(this, arguments); }

var prefix = _carbonComponents.settings.prefix;

var findInputNode = function findInputNode(wrapper) {
  return wrapper.find(".".concat(prefix, "--text-input"));
};

var openMenu = function openMenu(wrapper) {
  wrapper.find("[role=\"combobox\"]").simulate('click');
};

describe('ComboBox', function () {
  var mockProps;
  beforeEach(function () {
    mockProps = {
      id: 'test-combobox',
      items: (0, _testHelpers.generateItems)(5, _testHelpers.generateGenericItem),
      onChange: jest.fn(),
      placeholder: 'Filter...',
      type: 'default'
    };
  });
  it('should display the placeholder text when no items are selected and the control is not focused', function () {
    var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, mockProps));
    expect(findInputNode(wrapper).prop('value')).toBe('');
    expect(findInputNode(wrapper).prop('placeholder')).toBe(mockProps.placeholder);
  });
  it('should display the menu of items when a user clicks on the input', function () {
    var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, mockProps));
    findInputNode(wrapper).simulate('click');
    (0, _testHelpers.assertMenuOpen)(wrapper, mockProps);
  });
  it('should call `onChange` each time an item is selected', function () {
    var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, mockProps));
    expect(mockProps.onChange).not.toHaveBeenCalled();

    for (var i = 0; i < mockProps.items.length; i++) {
      openMenu(wrapper);
      wrapper.find('ForwardRef(ListBoxMenuItem)').at(i).simulate('click');
      expect(mockProps.onChange).toHaveBeenCalledTimes(i + 1);
      expect(mockProps.onChange).toHaveBeenCalledWith({
        selectedItem: mockProps.items[i]
      });
    }
  });
  it('capture filter text events', function () {
    var onInputChange = jest.fn();
    var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, _extends({}, mockProps, {
      onInputChange: onInputChange
    })));
    findInputNode(wrapper).simulate('change', {
      target: {
        value: 'something'
      }
    });
    expect(onInputChange).toHaveBeenCalledWith('something');
  });
  it('should render custom item components', function () {
    var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, mockProps));
    wrapper.setProps({
      itemToElement: function itemToElement(item) {
        return /*#__PURE__*/_react.default.createElement("div", {
          className: "mock-item"
        }, item.text);
      }
    });
    openMenu(wrapper);
    expect(wrapper.find(".mock-item").length).toBe(mockProps.items.length);
  });
  it('should let the user select an option by clicking on the option node', function () {
    var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, mockProps));
    openMenu(wrapper);
    wrapper.find('ForwardRef(ListBoxMenuItem)').at(0).simulate('click');
    expect(mockProps.onChange).toHaveBeenCalledTimes(1);
    expect(mockProps.onChange).toHaveBeenCalledWith({
      selectedItem: mockProps.items[0]
    });
    (0, _testHelpers.assertMenuClosed)(wrapper);
    mockProps.onChange.mockClear();
    openMenu(wrapper);
    wrapper.find('ForwardRef(ListBoxMenuItem)').at(1).simulate('click');
    expect(mockProps.onChange).toHaveBeenCalledTimes(1);
    expect(mockProps.onChange).toHaveBeenCalledWith({
      selectedItem: mockProps.items[1]
    });
  });
  describe('should display initially selected item found in `initialSelectedItem`', function () {
    it('using an object type for the `initialSelectedItem` prop', function () {
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, _extends({}, mockProps, {
        initialSelectedItem: mockProps.items[0]
      })));
      expect(findInputNode(wrapper).prop('value')).toEqual(mockProps.items[0].label);
    });
    it('using a string type for the `initialSelectedItem` prop', function () {
      // Replace the 'items' property in mockProps with a list of strings
      mockProps = _objectSpread(_objectSpread({}, mockProps), {}, {
        items: ['1', '2', '3']
      });
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, _extends({}, mockProps, {
        initialSelectedItem: mockProps.items[1]
      })));
      expect(findInputNode(wrapper).prop('value')).toEqual(mockProps.items[1]);
    });
  });
  describe('should display selected item found in `selectedItem`', function () {
    it('using an object type for the `selectedItem` prop', function () {
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, _extends({}, mockProps, {
        selectedItem: mockProps.items[0]
      })));
      expect(findInputNode(wrapper).prop('value')).toEqual(mockProps.items[0].label);
    });
    it('using a string type for the `selectedItem` prop', function () {
      // Replace the 'items' property in mockProps with a list of strings
      mockProps = _objectSpread(_objectSpread({}, mockProps), {}, {
        items: ['1', '2', '3']
      });
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, _extends({}, mockProps, {
        selectedItem: mockProps.items[1]
      })));
      expect(findInputNode(wrapper).prop('value')).toEqual(mockProps.items[1]);
    });
  });
  describe('when disabled', function () {
    it('should not let the user edit the input node', function () {
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, _extends({}, mockProps, {
        disabled: true
      })));
      expect(findInputNode(wrapper).prop('disabled')).toBe(true);
      expect(findInputNode(wrapper).prop('value')).toBe('');
      findInputNode(wrapper).simulate('change', {
        target: {
          value: 'a'
        }
      });
      expect(findInputNode(wrapper).prop('value')).toBe('');
    });
    it('should not let the user expand the menu', function () {
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, _extends({}, mockProps, {
        disabled: true
      })));
      openMenu(wrapper);
      expect((0, _testHelpers.findListBoxNode)(wrapper).hasClass('bx--list-box--expanded')).toBe(false);
    });
  });
  describe('downshift quirks', function () {
    it('should not trigger the menu when typing a space in input', function () {
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, mockProps));
      openMenu(wrapper);
      findInputNode(wrapper).simulate('change', {
        target: {
          value: ' '
        }
      });
      expect((0, _testHelpers.findMenuNode)(wrapper).length).toBe(1);
    });
    it('should set `inputValue` to an empty string if a false-y value is given', function () {
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_ComboBox.default, mockProps));
      expect(wrapper.find('input').instance().value).toBe('');
    });
  });
});