/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
/**
 * Notes: this file covers the three (as of this writing) subcommands
 * of apply:
 *  - view-last-applied
 *  - set-last-applied
 *  - edit-last-applied
 *
 */
import { i18n } from '@kui-shell/core';
import { flags } from './flags';
import { doExecWithPty } from './exec';
import { isUsage, doHelp } from '../../lib/util/help';
import { getCommandFromArgs } from '../../lib/util/util';
import { doGetAsMMR as getTransformer } from './get';
import { editSpec, formatToolbarText } from './edit';
import { getNamespaceForArgv } from './options';
import { isKubeResource } from '../../lib/model/resource';
import { hasLastApplied, mode as lastAppliedMode, label as lastAppliedModeLabel, order as lastAppliedOrder, renderLastApplied } from '../../lib/view/modes/last-applied';
const strings = i18n('plugin-kubectl');
/** View Transformer for view-last-applied and edit-last-applied */
export const viewLastApplied = (subcommand) => (args, response) => __awaiter(void 0, void 0, void 0, function* () {
    if (isKubeResource(response)) {
        if (hasLastApplied(response)) {
            const baseView = yield getTransformer(args, response);
            if (subcommand === 'view-last-applied') {
                return Object.assign(baseView, { defaultMode: lastAppliedMode });
            }
            else {
                const { content, contentType } = yield renderLastApplied(args.tab, response);
                const spec = editSpec(getCommandFromArgs(args), response.metadata.namespace, args, response, 'set-last-applied');
                const editMode = {
                    mode: lastAppliedMode,
                    label: lastAppliedModeLabel,
                    order: lastAppliedOrder - 1,
                    content,
                    contentType,
                    spec
                };
                return Object.assign(baseView, {
                    modes: [editMode],
                    defaultMode: lastAppliedMode,
                    toolbarText: formatToolbarText(args, response)
                });
            }
        }
        else {
            const error = new Error(strings('This resource has no last applied configuration'));
            error.code = 404;
            throw error;
        }
    }
});
function get(subcommand, args) {
    const command = getCommandFromArgs(args);
    if (isUsage(args)) {
        // special case: --help/-h
        return doHelp(command === 'k' ? 'kubectl' : command, args);
    }
    const idx = args.argvNoOptions.indexOf(subcommand);
    const kind = args.argvNoOptions[idx + 1] || '';
    const name = args.argvNoOptions[idx + 2] || '';
    const file = args.parsedOptions.f || args.parsedOptions.filename;
    if (file) {
        return args.REPL.qexec(`${command} get -f ${file} ${getNamespaceForArgv(args)} -o yaml`);
    }
    else {
        return args.REPL.qexec(`${command} get ${kind} ${name} ${getNamespaceForArgv(args)} -o yaml`);
    }
}
function withTransformer(viewTransformer) {
    return Object.assign({}, flags, { viewTransformer });
}
export function registerApplySubcommands(registrar, cmd) {
    registrar.listen(`/${cmd}/apply/view-last-applied`, get.bind(undefined, 'view-last-applied'), withTransformer(viewLastApplied('view-last-applied')));
    registrar.listen(`/${cmd}/apply/edit-last-applied`, get.bind(undefined, 'edit-last-applied'), withTransformer(viewLastApplied('edit-last-applied')));
    registrar.listen(`/${cmd}/apply/set-last-applied`, doExecWithPty);
}
export default function registerForKubectl(registrar) {
    registerApplySubcommands(registrar, 'kubectl');
    registerApplySubcommands(registrar, 'k');
}
//# sourceMappingURL=apply-subcommands.js.map