"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.wireToTabEvents = wireToTabEvents;
exports.unwireToTabEvents = unwireToTabEvents;
exports.wireToStandardEvents = wireToStandardEvents;
exports.unwireToStandardEvents = unwireToStandardEvents;
exports.eventBus = exports.default = void 0;

var _events = require("events");

var _command = require("../models/command");

var _tab = require("../webapp/tab");

/*
 * Copyright 2017 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* eslint-disable no-dupe-class-members */
const eventChannelUnsafe = new _events.EventEmitter();
eventChannelUnsafe.setMaxListeners(100);
var _default = eventChannelUnsafe;
exports.default = _default;

class EventBusBase {
  constructor() {
    const eventBus = new _events.EventEmitter();
    eventBus.setMaxListeners(100);
    this.eventBus = eventBus;
  }

}

class WriteEventBus extends EventBusBase {
  emit(channel, args) {
    return this.eventBus.emit(channel, args);
  }
  /** User switching focus from one Split to another, within one Tab */


  emitSplitSwitch(tab) {
    this.eventBus.emit('/tab/switch/split', tab);
  }

  emitTabLayoutChange(tabUUID, evt = {
    isSidecarNowHidden: false,
    isWidthConstrained: false
  }) {
    setTimeout(() => this.eventBus.emit(`/tab/layout/change/${tabUUID}`, evt));
  }

  emitEnvUpdate(key, value) {
    this.eventBus.emit(`/env/update/${key}`, value);
  }

  emitCommandEvent(which, event, isReplay) {
    this.eventBus.emit(`/command/${which}`, event);

    if (event.execType !== _command.ExecType.Nested) {
      const from = isReplay ? 'replay' : 'fromuser';
      this.eventBus.emit(`/command/${which}/${from}`, event);
      this.eventBus.emit(`/command/${which}/${from}/${event.tab.uuid}`, event);
      const primary = (0, _tab.getPrimaryTabId)(event.tab);

      if (event.tab.uuid !== primary) {
        this.eventBus.emit(`/command/${which}/${from}/${primary}`, event);
      }

      this.eventBus.emit(`/command/${which}/${from}/${primary}/type/${event.execType}`, event);
    }
  }

  emitCommandStart(event, isReplay = false) {
    this.emitCommandEvent('start', event, isReplay);
  }

  emitCommandComplete(event, isReplay = false) {
    this.emitCommandEvent('complete', event, isReplay);

    if (event.execType !== _command.ExecType.Nested) {
      const from = isReplay ? 'replay' : 'fromuser';
      this.eventBus.emit(`/command/complete/${from}/${event.responseType}`, event);
      this.eventBus.emit(`/command/complete/${from}/${event.responseType}/${event.tab.uuid}`, event);
      const primary = (0, _tab.getPrimaryTabId)(event.tab);

      if (primary !== event.tab.uuid) {
        this.eventBus.emit(`/command/complete/${from}/${event.responseType}/${primary}`, event);
      }
    }
  }
  /** Request a Snapshot of the given Tab */


  emitSnapshotRequest(evt, tabId) {
    this.eventBus.emit(`/snapshot/request/${tabId}`, evt);
  }
  /** Request a status stripe change */


  emitStatusStripeChangeRequest(evt) {
    this.eventBus.emit('/status-stripe/change', evt);
  }

  emitWithTabId(channel, tabId, tab) {
    this.eventBus.emit(`${channel}/${tabId}`, tabId, tab);
  }

}

class ReadEventBus extends WriteEventBus {
  on(channel, listener) {
    return this.eventBus.on(channel, listener);
  }

  off(channel, listener) {
    return this.eventBus.off(channel, listener);
  }

  onRemoveSnapshotable(listener) {
    this.eventBus.on('/snapshot/element/remove', listener);
  }
  /** Snapshot the Block state of the given Tab */


  onSnapshotRequest(listener, tabId) {
    this.eventBus.on(`/snapshot/request/${tabId}`, listener);
  }
  /** Snapshot the Block state of the given Tab */


  offSnapshotRequest(listener, tabId) {
    this.eventBus.off(`/snapshot/${tabId}`, listener);
  }
  /** Request a status stripe change */


  onStatusStripeChangeRequest(listener) {
    this.eventBus.on('/status-stripe/change', listener);
  }
  /** Request a status stripe change */


  offStatusStripeChangeRequest(listener) {
    this.eventBus.off('/status-stripe/change', listener);
  }
  /** User switching focus from one Split to another, within one Tab */


  onSplitSwitch(listener) {
    this.eventBus.on('/tab/switch/split', listener);
  }
  /** User switching focus from one Split to another, within one Tab */


  offSplitSwitch(listener) {
    this.eventBus.off('/tab/switch/split', listener);
  }

  onTabLayoutChange(tabUUID, listener) {
    this.eventBus.on(`/tab/layout/change/${tabUUID}`, listener);
  }

  offTabLayoutChange(tabUUID, listener) {
    this.eventBus.off(`/tab/layout/change/${tabUUID}`, listener);
  }

  onEnvUpdate(key, listener) {
    this.eventBus.on(`/env/update/${key}`, listener);
  }

  offEnvUpdate(key, listener) {
    this.eventBus.off(`/env/update/${key}`, listener);
  }

  onCommand(which, splitId, splitHandler, onReplay = false) {
    if (onReplay) {
      this.eventBus.on(`/command/${which}/replay/${splitId}`, splitHandler);
    } else {
      this.eventBus.on(`/command/${which}/fromuser/${splitId}`, splitHandler);
    }
  }

  offCommand(which, splitId, splitHandler) {
    this.eventBus.off(`/command/${which}/fromuser/${splitId}`, splitHandler);
  }

  onAnyCommandStart(handler) {
    this.eventBus.on('/command/start/fromuser', handler);
  }

  offAnyCommandStart(handler) {
    this.eventBus.off('/command/start/fromuser', handler);
  }

  onAnyCommandComplete(handler) {
    this.eventBus.on('/command/complete/fromuser', handler);
  }

  offAnyCommandComplete(handler) {
    this.eventBus.off('/command/complete/fromuser', handler);
  }

  onCommandStart(splitId, splitHandler, onReplay = false) {
    this.onCommand('start', splitId, splitHandler, onReplay);
  }

  onceCommandStarts(tabId, handler) {
    this.eventBus.once(`/command/start/fromuser/${tabId}`, handler);
  }

  onResponseType(responseType, splitId, splitHandler, onReplay = true) {
    this.eventBus.on(`/command/complete/fromuser/${responseType}/${splitId}`, splitHandler); // if you don't want the sidecar to open on replay, comment this out:

    if (onReplay) {
      this.eventBus.on(`/command/complete/replay/${responseType}/${splitId}`, splitHandler);
    }
  }

  offResponseType(responseType, splitId, splitHandler) {
    this.eventBus.off(`/command/complete/fromuser/${responseType}/${splitId}`, splitHandler);
  }

  onScalarResponse(splitId, splitHandler) {
    this.onResponseType('ScalarResponse', splitId, splitHandler);
  }

  offScalarResponse(splitId, splitHandler) {
    this.offResponseType('ScalarResponse', splitId, splitHandler);
  }

  onMultiModalResponse(tabId, handler, onReplay) {
    this.onResponseType('MultiModalResponse', tabId, handler, onReplay);
  }

  offMultiModalResponse(tabId, handler) {
    this.offResponseType('MultiModalResponse', tabId, handler);
  }

  onNavResponse(tabId, handler, onReplay) {
    this.onResponseType('NavResponse', tabId, handler, onReplay);
  }

  offNavResponse(tabId, handler) {
    this.offResponseType('NavResponse', tabId, handler);
  }

  onCommandComplete(splitId, splitHandler, onReplay = false) {
    this.onCommand('complete', splitId, splitHandler, onReplay);
  }

  offCommandStart(splitId, splitHandler) {
    this.offCommand('start', splitId, splitHandler);
  }

  offCommandComplete(splitId, splitHandler) {
    this.offCommand('complete', splitId, splitHandler);
  }

  onWithTabId(channel, tabId, listener) {
    this.eventBus.on(`${channel}/${tabId}`, listener);
  }

  offWithTabId(channel, tabId, listener) {
    this.eventBus.off(`${channel}/${tabId}`, listener);
  }

  onceWithTabId(channel, tabId, listener) {
    this.eventBus.once(`${channel}/${tabId}`, listener);
  }

  once(channel, listener) {
    return this.eventBus.once(channel, listener);
  }

}

class EventBus extends ReadEventBus {}

const eventBus = new EventBus();
/**
 * Hook an event listener up to tab events.
 *
 */

exports.eventBus = eventBus;

function wireToTabEvents(listener) {
  eventBus.on('/tab/new', listener);
  eventBus.on('/tab/switch/request/done', listener);
  eventBus.onSplitSwitch(listener);
}
/**
 * Unhook
 *
 */


function unwireToTabEvents(listener) {
  eventBus.off('/tab/new', listener);
  eventBus.off('/tab/switch/request/done', listener);
  eventBus.offSplitSwitch(listener);
}
/**
 * Hook an event listener up to the family of standard user
 * interaction events.
 *
 */


function wireToStandardEvents(listener) {
  wireToTabEvents(listener);
  eventBus.onAnyCommandComplete(listener);
  eventChannelUnsafe.on('/terminal/clear', listener);
}
/**
 * Unhook
 *
 */


function unwireToStandardEvents(listener) {
  unwireToTabEvents(listener);
  eventBus.offAnyCommandComplete(listener);
  eventChannelUnsafe.off('/terminal/clear', listener);
}