/*
 * Copyright 2019 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import Debug from 'debug';
import { eventBus } from '../core/events';
import { promiseEach } from '../util/async';
const debug = Debug('core/models/TabState');
const registrar = [];
export function registerTabState(registration) {
    registrar.push(registration);
}
/**
 * State that we want to keep per tab
 *
 */
export default class TabState {
    constructor(uuid, _desiredStatusStripeDecoration = { type: 'default' }, _parent) {
        this.uuid = uuid;
        this._desiredStatusStripeDecoration = _desiredStatusStripeDecoration;
        this._parent = _parent;
        /** is the tab ready for command execution? */
        this.ready = false;
        /** state map
         * outer key is `TabStateRegistrar.name`, inner key is `TabStateRegistrar.apiVersion`
         * e.g. { 'plugins/plugin-core': {'v1': {'cwd': '/'}}}
         */
        this._state = {};
        /** functions to capture the states of tab */
        this.captures = [];
        /** functions to restore the states of the tab */
        this.restores = [];
        /** functions to capture this tab state and restore another tab state */
        this.switchTos = [];
        registrar.forEach(_ => {
            this.register(_.name, _.apiVersion, _.capture, _.restore, _.switchTo);
        });
    }
    get state() {
        return this._state;
    }
    checkExistence(name, apiVersion) {
        if (!this.state[name]) {
            throw new Error(`${name} doesn't exist in tab state`);
        }
        else if (!this.state[name][apiVersion]) {
            throw new Error(`${name} doesn't have version ${apiVersion}`);
        }
        else {
            return true;
        }
    }
    register(name, apiVersion, capture, restore, switchTo) {
        // initialize the state
        if (!this.state[name]) {
            this.state[name] = { [apiVersion]: {} };
        }
        else if (!this.state[name][apiVersion]) {
            this._state[name][apiVersion] = {};
        }
        else {
            throw new Error(`${name} ${apiVersion} already registered`);
        }
        this.captures.push(capture);
        this.restores.push(restore);
        this.switchTos.push(switchTo);
    }
    getState(name, apiVersion, key) {
        if (this.checkExistence(name, apiVersion)) {
            if (this.state[name][apiVersion][key] === undefined) {
                throw new Error(`${name} ${apiVersion} doesn't have state ${key}`);
            }
            else {
                return this.state[name][apiVersion][key];
            }
        }
    }
    setState(name, apiVersion, key, value) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.checkExistence(name, apiVersion)) {
                this.state[name][apiVersion][key] = value;
            }
        });
    }
    /** Capture contextual global state */
    capture() {
        this.captures.forEach(capture => capture(this));
    }
    /** Capture contextual global state and then restore `nextTabState` */
    switchTo(nextTabState) {
        return __awaiter(this, void 0, void 0, function* () {
            yield promiseEach(this.switchTos, (switchTo) => __awaiter(this, void 0, void 0, function* () {
                yield switchTo(this, nextTabState);
            }));
            nextTabState.updateStatusStripe();
        });
    }
    /** Clone the captured state */
    cloneWithUUID(uuid) {
        this.capture(); // keep the function name, and mean capture all
        const clone = new TabState(uuid, this.desiredStatusStripeDecoration, this);
        clone.capture();
        debug('cloned tab state', clone.uuid, clone._state);
        return clone;
    }
    /** Enforce our desired status stripe decorations */
    updateStatusStripe() {
        if (this._parent) {
            this._parent.updateStatusStripe();
        }
        else {
            eventBus.emitStatusStripeChangeRequest(this.desiredStatusStripeDecoration);
        }
    }
    get desiredStatusStripeDecoration() {
        return this._desiredStatusStripeDecoration;
    }
    set desiredStatusStripeDecoration(decor) {
        this._desiredStatusStripeDecoration = decor;
        if (this._parent) {
            this._parent.desiredStatusStripeDecoration = decor;
        }
        else {
            this.updateStatusStripe();
        }
    }
    /**
     * Restore tab state
     *
     */
    restore() {
        return __awaiter(this, void 0, void 0, function* () {
            this.restores.forEach(restore => restore(this));
            this.updateStatusStripe();
        });
    }
}
//# sourceMappingURL=tab-state.js.map