import { ReturnValue } from './return-value';
export declare class PromisePoolExecutor<T, R> {
    /**
     * The list of items to process.
     */
    private items;
    /**
     * The number of concurrently running tasks.
     */
    private concurrency;
    /**
     * The intermediate list of currently running tasks.
     */
    private readonly tasks;
    /**
     * The list of results.
     */
    private readonly results;
    /**
     * The async processing function receiving each item from the `items` array.
     */
    private handler;
    /**
     * The async error handling function.
     */
    private errorHandler?;
    /**
     * The list of errors.
     */
    private readonly errors;
    /**
     * Creates a new promise pool executer instance with a default concurrency of 10.
     */
    constructor();
    /**
     * Set the number of tasks to process concurrently the promise pool.
     *
     * @param {Integer} concurrency
     *
     * @returns {PromisePoolExecutor}
     */
    withConcurrency(concurrency: number): this;
    /**
     * Set the items to be processed in the promise pool.
     *
     * @param {Array} items
     *
     * @returns {PromisePoolExecutor}
     */
    for(items: T[]): this;
    /**
     * Set the handler that is applied to each item.
     *
     * @param {Function} action
     *
     * @returns {PromisePoolExecutor}
     */
    withHandler(action: (item: T) => R | Promise<R>): this;
    /**
     * Set the error handler function to execute when an error occurs.
     *
     * @param {Function} handler
     *
     * @returns {PromisePoolExecutor}
     */
    handleError(handler?: (error: Error, item: T) => Promise<void> | void): this;
    /**
     * Determines whether the number of active tasks is greater or equal to the concurrency limit.
     *
     * @returns {Boolean}
     */
    hasReachedConcurrencyLimit(): boolean;
    /**
     * Returns the number of active tasks.
     *
     * @returns {Number}
     */
    activeCount(): number;
    /**
     * Start processing the promise pool.
     *
     * @returns {Array}
     */
    start(): Promise<ReturnValue<T, R>>;
    /**
     * Ensure valid inputs and throw otherwise.
     *
     * @throws
     */
    validateInputs(): void;
    /**
     * Starts processing the promise pool by iterating over the items
     * and running each item through the async `callback` function.
     *
     * @param {Function} callback
     *
     * @returns {Promise}
     */
    process(): Promise<ReturnValue<T, R>>;
    /**
     * Creates a deferred promise and pushes the related callback to the pending
     * queue. Returns the promise which is used to wait for the callback.
     *
     * @returns {Promise}
     */
    processingSlot(): Promise<void>;
    /**
     * Wait for one of the active tasks to finish processing.
     */
    waitForTaskToFinish(): Promise<void>;
    /**
     * Create a processing function for the given `item`.
     *
     * @param {*} item
     */
    startProcessing(item: T): void;
    /**
     * Ensures a returned promise for the processing of the given `item`.
     *
     * @param item
     *
     * @returns {*}
     */
    createTaskFor(item: T): Promise<any>;
    /**
     * Wait for all active tasks to finish. Once all the tasks finished
     * processing, returns an object containing the results and errors.
     *
     * @returns {Object}
     */
    drained(): Promise<ReturnValue<T, R>>;
    /**
     * Wait for all of the active tasks to finish processing.
     */
    drainActiveTasks(): Promise<void>;
}
