"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mount = mount;
exports.absolute = absolute;
exports.findMatchingMounts = findMatchingMounts;
exports.findMount = findMount;
exports.multiFindMount = multiFindMount;
exports.isLocal = isLocal;

var _slash = _interopRequireDefault(require("slash"));

var _path = require("path");

var _core = require("@kui-shell/core");

var _initDone = require("./initDone");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};

/**
 * Current VFS mounts
 *
 */
const _currentMounts = [];
/**
 * Establish a total ordering of mounts, based on depth in the
 * tree. Sibling order is arbitrary, at the moment.
 *
 */

function orient(A, B) {
  const a = A.mountPath.split(/\//).filter(_ => _);
  const b = B.mountPath.split(/\//).filter(_ => _);
  let lastCommonIdx = 0;
  const N = Math.min(a.length, b.length);

  while (++lastCommonIdx < N) {
    if (a[lastCommonIdx] !== b[lastCommonIdx]) {
      break;
    }
  }

  if (lastCommonIdx === 0 || a.length === b.length) {
    return 0;
  } else if (lastCommonIdx === a.length) {
    return -1;
  } else {
    return b.length - a.length;
  }
}
/** Low-level mount */


function _mount(vfs) {
  const existingIdx = _currentMounts.findIndex(mount => mount.mountPath === vfs.mountPath);

  if (existingIdx >= 0) {
    // replace existing mount
    _currentMounts.splice(existingIdx, 1, vfs);
  } else {
    // add new mount
    _currentMounts.push(vfs);

    _currentMounts.sort(orient);
  }
}
/** Invoke the given VFS-producing function, and mount all of the resulting VFS's */


function mountAll({
  REPL
}, vfsFn) {
  return __awaiter(this, void 0, void 0, function* () {
    const mounts = yield vfsFn(REPL);
    const A = Array.isArray(mounts) ? mounts : [mounts];
    yield Promise.all(A.map(_mount));
  });
}
/**
 * Mount a VFS
 *
 */


function mount(vfs, placeholderMountPath) {
  if (typeof vfs !== 'function') {
    _mount(vfs);
  } else {
    (0, _initDone.pushInitDone)( // eslint-disable-next-line no-async-promise-executor
    new Promise((resolve, reject) => __awaiter(this, void 0, void 0, function* () {
      const tab = (0, _core.getCurrentTab)();

      if (!tab) {
        try {
          let debounce = false;

          _core.eventBus.on('/tab/new', tab => __awaiter(this, void 0, void 0, function* () {
            try {
              if (!debounce) {
                debounce = true;
                yield mountAll(tab, vfs);
                resolve(undefined);
              }
            } catch (err) {
              console.error('Error in mount 1', err);
              reject(err);
            }
          }));
        } catch (err) {
          console.error('Error in mount 2', err);
          reject(err);
        }
      } else {
        try {
          yield mountAll(tab, vfs);
          resolve(undefined);
        } catch (err) {
          console.error('Error in mount 3', err);
          reject(err);
        }
      }
    })), placeholderMountPath);
  }
}
/** @return the absolute path to `filepath` */


function absolute(filepath) {
  return (0, _path.isAbsolute)((0, _core.expandHomeDir)(filepath.toString())) ? filepath : (0, _path.join)((0, _core.cwd)(), filepath.toString());
}
/** Lookup compiatible matching mount */


function findMatchingMounts(filepath, checkClient = false) {
  const isClient = (0, _core.inBrowser)();
  filepath = absolute(filepath);

  const mounts = _currentMounts.filter(mount => mount.mountPath.startsWith(filepath) && (!checkClient || !isClient || mount.isVirtual));

  return mounts;
}
/**
 * Lookup compiatible mount. Returns one of:
 * - a VFS mount, if it encloses the given filepath
 *
 * - true if `allowInner` and there exists a mount s.t. filepath
 *     encloses it (i.e. filepath is a parent of some mount)
 *
 * - otherwise, the local VFS mount
 */


function findMount(filepath, checkClient = false, allowInner = false) {
  return __awaiter(this, void 0, void 0, function* () {
    yield (0, _initDone.waitForMountsToFinish)(filepath);
    const isClient = (0, _core.inBrowser)();
    filepath = absolute(filepath); // filepath: /kuifake   Possibilities limited to [/kuifake]
    // mounts: ['/kuifake/fake1', '/tmpo', '/kuifake/fake2', '/kui', '/']
    // first loop should find nothing
    // if allowInner, second loop should return true; done!
    // filepath: /kuifake/fake1  Possibilities limited to [/kuifake, /kuifake/fake1]
    // mounts: ['/kuifake/fake1', '/tmpo', '/kuifake/fake2', '/kui', '/']
    // first loop should find /kuifake/fake1; done!
    // filepath: /kuifake/fake1/E1/f1  Possibilities limited to [/kuifake, /kuifake/fake1, /kuifake/fake1/E1, /kuifake/fake1/E1/f1]
    // mounts: ['/kuifake/fake1', '/tmpo', '/kuifake/fake2', '/kui', '/']
    // first loop should find /kuifake/fake1; done!
    // filepath: /a/b/c/d
    // mounts: ['/kuifake/fake1', '/tmpo', '/kuifake/fake2', '/kui', '/']
    // first loop should find '/'; done!
    // This loop checks if any mount **encloses** the given filepath (hence startsWith())
    // Important: search from longest to shortest!! e.g. we don't want
    // /aaa/b/c to match a mount /a when we also have a mount /aaa/b

    const splitPath = filepath.split(/\//);
    const possibilities = [];

    for (let idx = 1; idx < splitPath.length; idx++) {
      if (splitPath[idx]) {
        possibilities.push('/' + splitPath.slice(1, idx + 1).join('/'));
      }
    }

    let foundMount;

    for (let idx = 0; idx < _currentMounts.length; idx++) {
      const mount = _currentMounts[idx];
      const {
        mountPath
      } = mount;

      if (possibilities.includes(mountPath) && (!checkClient || !isClient || mount.isVirtual)) {
        foundMount = mount;
        break;
      }
    }

    if (!foundMount) {
      if (allowInner) {
        // ok, then look for a mount where the given filepath encloses the mount
        for (let idx = 0; idx < _currentMounts.length; idx++) {
          const mount = _currentMounts[idx];

          if (mount.mountPath.startsWith(filepath) && (!checkClient || !isClient || mount.isVirtual)) {
            return;
          }
        }
      } // local fallback; see https://github.com/IBM/kui/issues/5898


      foundMount = _currentMounts.find(mount => mount.isLocal);
    } // debug(`findMount ${filepath}->${foundMount.mountPath}`)


    return foundMount;
  });
}
/** Lookup compatible mounts */


function multiFindMount(filepaths, checkClient = false) {
  return __awaiter(this, void 0, void 0, function* () {
    if (filepaths.length === 0) {
      return multiFindMount([(0, _core.cwd)()], checkClient);
    }

    return (yield Promise.all(filepaths.map(filepath => __awaiter(this, void 0, void 0, function* () {
      return {
        filepaths: [(0, _slash.default)(filepath)],
        mount: yield findMount(filepath, checkClient)
      };
    })))).filter(_ => _.mount !== undefined).reduce((mounts, mount) => {
      if (mounts.length === 0 || mounts[mounts.length - 1].mount.mountPath !== mount.mount.mountPath) {
        mounts.push(mount);
      } else {
        mounts[mounts.length - 1].filepaths.splice(mounts.length, 0, ...mount.filepaths);
      }

      return mounts;
    }, []);
  });
}
/**
 * Does the filepath specify a local file?
 *
 */


function isLocal(filepath) {
  return __awaiter(this, void 0, void 0, function* () {
    const mount = yield findMount(filepath);
    return mount && mount.isLocal;
  });
}