/// <reference types="node" />
import { Arguments, ParsedOptions, REPL, Table } from '@kui-shell/core';
import { FStat } from '../lib/fstat';
import { KuiGlobOptions, GlobStats } from '../lib/glob';
declare type DirEntry = GlobStats;
export { DirEntry };
export interface ParallelismOptions extends ParsedOptions {
    /** Parallelism */
    P: number;
    /** Memory per task */
    m: string;
    memory: string;
}
/**
 * A Virtual File System implements `ls`, `cp`, etc. Filesystem
 * operations against filepaths that match this `mountPath` as a
 * prefix will be delegated to this `VFS` impl.
 *
 */
export interface VFS {
    /** Path to mount point */
    mountPath: string;
    /** Is this a local mount? */
    isLocal: boolean;
    /** Is this a virtual mount? i.e. one that works in a browser without server-side proxy support */
    isVirtual: boolean;
    /** Any tags that the provider might want to associate with their VFS */
    tags?: string[];
    /** Directory listing */
    ls(opts: Pick<Arguments<KuiGlobOptions>, 'tab' | 'REPL' | 'parsedOptions'>, filepaths: string[]): DirEntry[] | Promise<DirEntry[]>;
    /** Insert filepath into directory */
    cp(opts: Pick<Arguments, 'command' | 'REPL' | 'parsedOptions' | 'execOptions'>, srcFilepaths: string[], dstFilepath: string, srcIsSelf: boolean[], dstIsSelf: boolean, srcProvider: VFS[], dstProvider: VFS): Promise<string | Table | true | (string | Table)[]>;
    /** Remove filepath */
    rm(opts: Pick<Arguments, 'command' | 'REPL' | 'parsedOptions' | 'execOptions'>, filepath: string, recursive?: boolean): Promise<string | boolean>;
    /** Fetch contents */
    fstat(opts: Pick<Arguments, 'REPL' | 'parsedOptions'>, filepath: string, withData?: boolean, enoentOk?: boolean): Promise<FStat>;
    /** Fetch content slice */
    fslice(filename: string, offset: number, length: number): Promise<string>;
    /** write data to file */
    fwrite(opts: Pick<Arguments, 'REPL'>, fullpath: string, data: string | Buffer): Promise<void>;
    /** Create a directory/bucket */
    mkdir(opts: Pick<Arguments, 'command' | 'REPL' | 'argvNoOptions' | 'parsedOptions' | 'execOptions'>, filepath: string): Promise<void>;
    /** remove a directory/bucket */
    rmdir(opts: Pick<Arguments, 'command' | 'REPL' | 'parsedOptions' | 'execOptions'>, filepath: string): Promise<void>;
}
declare type VFSProducingFunction = (repl: REPL) => VFS | VFS[] | Promise<VFS | VFS[]>;
/**
 * Mount a VFS
 *
 */
export declare function mount(vfs: VFS | VFSProducingFunction, placeholderMountPath?: string): void;
/** @return the absolute path to `filepath` */
export declare function absolute(filepath: string): string;
/** Lookup compiatible matching mount */
export declare function findMatchingMounts(filepath: string, checkClient?: boolean): VFS[];
/**
 * Lookup compiatible mount. Returns one of:
 * - a VFS mount, if it encloses the given filepath
 *
 * - true if `allowInner` and there exists a mount s.t. filepath
 *     encloses it (i.e. filepath is a parent of some mount)
 *
 * - otherwise, the local VFS mount
 */
export declare function findMount(filepath: string, checkClient?: boolean, allowInner?: boolean): Promise<VFS>;
/** Lookup compatible mounts */
export declare function multiFindMount(filepaths: string[], checkClient?: boolean): Promise<{
    filepaths: string[];
    mount: VFS;
}[]>;
/**
 * Does the filepath specify a local file?
 *
 */
export declare function isLocal(filepath: string): Promise<boolean>;
