/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { createGunzip } from 'zlib';
import { createReadStream } from 'fs';
import { expandHomeDir } from '@kui-shell/core';
import { mount } from '.';
import { kuiglob } from '../lib/glob';
import { fstat } from '../lib/fstat';
import { _fwrite } from '../lib/fwrite';
class LocalVFS {
    constructor() {
        this.mountPath = '/';
        this.isLocal = true;
        this.isVirtual = false;
    }
    ls(opts, filepaths) {
        return __awaiter(this, void 0, void 0, function* () {
            return kuiglob({
                tab: opts.tab,
                argvNoOptions: ['kuiglob', ...filepaths],
                parsedOptions: opts.parsedOptions
            });
        });
    }
    /** Insert filepath into directory */
    cp(opts) {
        return __awaiter(this, void 0, void 0, function* () {
            return opts.REPL.qexec(`sendtopty ${opts.command.replace(/^vfs/, '')}`, undefined, undefined, Object.assign(opts.execOptions, { quiet: opts.parsedOptions.i ? false : opts.execOptions.quiet }));
        });
    }
    /** Remove filepath */
    rm(opts) {
        return __awaiter(this, void 0, void 0, function* () {
            yield opts.REPL.qexec(`sendtopty ${opts.command.replace(/^vfs/, '')}`, undefined, undefined, Object.assign(opts.execOptions, { quiet: false }));
            return true;
        });
    }
    /** Fetch contents */
    fstat(opts, filepath, withData, enoentOk) {
        return __awaiter(this, void 0, void 0, function* () {
            return fstat({
                argvNoOptions: ['fstat', filepath],
                parsedOptions: { 'with-data': withData, 'enoent-ok': enoentOk }
            });
        });
    }
    /** Write data to a files */
    fwrite(opts, filepath, data) {
        return __awaiter(this, void 0, void 0, function* () {
            return _fwrite(filepath, data);
        });
    }
    /** Fetch content slice */
    fslice(filepath, offset, _length) {
        return __awaiter(this, void 0, void 0, function* () {
            const fullpath = expandHomeDir(filepath);
            return new Promise((resolve, reject) => {
                let data = '';
                // re: end = _length - 1, this is because the end option is inclusive and _length is not
                if (filepath.endsWith('.gz')) {
                    createReadStream(fullpath, { start: offset, end: _length ? offset + _length - 1 : Infinity })
                        .pipe(createGunzip())
                        .on('error', (err) => {
                        if (err.code === 'Z_BUF_ERROR') {
                            // this may happen when reading a part of a gzip file
                            resolve(data);
                        }
                        else {
                            reject(err);
                        }
                    })
                        .on('end', () => resolve(data))
                        .on('data', d => (data += d));
                }
                else {
                    createReadStream(fullpath, { start: offset, end: _length ? offset + _length - 1 : Infinity })
                        .on('error', reject)
                        .on('end', () => {
                        resolve(data);
                    })
                        .on('data', d => (data += d));
                }
            });
        });
    }
    /** Create a directory/bucket */
    mkdir(opts) {
        return __awaiter(this, void 0, void 0, function* () {
            yield opts.REPL.qexec(`sendtopty ${opts.command.replace(/^vfs/, '')}`, undefined, undefined, Object.assign(opts.execOptions, { quiet: false }));
        });
    }
    /** Remove a directory/bucket */
    rmdir(opts) {
        return __awaiter(this, void 0, void 0, function* () {
            yield opts.REPL.qexec(`sendtopty ${opts.command.replace(/^vfs/, '')}`, undefined, undefined, Object.assign(opts.execOptions, { quiet: false }));
        });
    }
}
export default () => __awaiter(void 0, void 0, void 0, function* () {
    mount(new LocalVFS());
});
//# sourceMappingURL=local.js.map