/*
 * Copyright 2019 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
/* eslint-disable no-async-promise-executor */
import Debug from 'debug';
import { eventBus, inBrowser, i18n } from '@kui-shell/core';
import { setOnline, setOffline } from './ui';
const strings = i18n('plugin-bash-like');
const debug = Debug('plugins/bash-like/pty/session');
/**
 * Return the cached websocket for the given tab
 *
 */
let _singleChannel; // share session across tabs see https://github.com/IBM/kui/issues/6453
let _exiting = false;
// eslint-disable-next-line @typescript-eslint/no-unused-vars
export function getChannelForTab(tab) {
    if (_exiting) {
        // prevent any stagglers re-establishing a channel
        throw new Error('Exiting');
    }
    else {
        return _singleChannel;
    }
}
/**
 * Set it
 *
 */
export function setChannelForTab(tab, channel) {
    debug('initializing pty channel: set');
    _singleChannel = channel;
    // listen for the end of the world
    channel.then(chan => {
        debug('initializing pty channel: success');
        window.addEventListener('beforeunload', () => {
            _exiting = true; // prevent any stagglers re-establishing a channel
            chan.close();
        });
    });
    return _singleChannel;
}
/**
 * Return the session for the given tab
 *
 */
export function getSessionForTab(tab) {
    debug('pty channel: get', tab.uuid);
    return _singleChannel;
}
/**
 * Keep trying until we can establish a session
 *
 */
export function pollUntilOnline(tab) {
    return new Promise(resolve => {
        let isOnline = false;
        const initialSetOffline = setTimeout(() => {
            if (!isOnline) {
                setOffline();
            }
        }, 5000);
        const once = (iter = 0) => {
            debug('trying to establish session', tab);
            return tab.REPL.qexec('echo initializing session', undefined, undefined, {
                tab
            })
                .then(() => {
                isOnline = true;
                clearTimeout(initialSetOffline);
                setOnline();
                resolve();
            })
                .catch(error => {
                const err = error;
                if (err.code !== 503) {
                    // don't bother complaining too much about connection refused
                    console.error('error establishing session', err.code, err.statusCode, err);
                    _singleChannel = undefined;
                }
                setTimeout(() => once(iter + 1), iter < 10 ? 2000 : iter < 100 ? 4000 : 10000);
                return strings('Could not establish a new session');
            });
        };
        once();
    });
}
/**
 * This function establishes a promise of a websocket channel for the
 * given tab
 *
 */
function newSessionForTab(tab) {
    return __awaiter(this, void 0, void 0, function* () {
        const thisSession = _singleChannel ||
            new Promise((resolve, reject) => __awaiter(this, void 0, void 0, function* () {
                try {
                    yield pollUntilOnline(tab);
                    tab.classList.add('kui--session-init-done');
                    resolve(yield getChannelForTab(tab));
                }
                catch (err) {
                    reject(err);
                }
            }));
        yield thisSession;
        tab.classList.add('kui--session-init-done');
    });
}
/** Connection to Kui proxy has been severed */
export function invalidateSession(tab) {
    _singleChannel = undefined;
    eventBus.emit('/tab/offline', tab);
    eventBus.emitWithTabId(`/tab/offline`, tab.state.uuid);
}
/**
 * Initialize per-tab websocket session management
 *
 */
export function init(registrar) {
    return __awaiter(this, void 0, void 0, function* () {
        const { proxyServer } = yield import('@kui-shell/client/config.d/proxy.json').catch(() => {
            console.log('using default proxy configuration');
            return { proxyServer: { enabled: false } };
        });
        if (inBrowser() && proxyServer.enabled !== false) {
            debug('initializing pty sessions');
            registrar.registerSessionInitializer(newSessionForTab);
        }
    });
}
//# sourceMappingURL=session.js.map