"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = plugin;
exports.tabButtonSelector = void 0;

var _core = require("@kui-shell/core");

/*
 * Copyright 2018 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};

// TODO fixme; this is needed by a few tests
const tabButtonSelector = '#new-tab-button';
exports.tabButtonSelector = tabButtonSelector;
const strings = (0, _core.i18n)('plugin-core-support');
const usage = {
  strict: 'switch',
  command: 'switch',
  required: [{
    name: 'tabIndex',
    numeric: true,
    docs: 'Switch to the given tab index'
  }]
};
/**
 * Close the current tab
 *
 */

function closeTab(tab, closeAllSplits) {
  const uuid = closeAllSplits ? (0, _core.getPrimaryTabId)(tab) : tab.uuid;

  _core.eventBus.emitWithTabId('/tab/close/request', uuid, tab);

  return true;
}
/** Load snapshot model from disk */


function loadSnapshotBuffer(REPL, filepath) {
  return __awaiter(this, void 0, void 0, function* () {
    return Buffer.from((yield REPL.rexec(`vfs fstat ${REPL.encodeComponent(filepath)} --with-data`)).content.data);
  });
}
/**
 * The goal here is to offer a simple command structure for managing tabs
 *
 */


function plugin(commandTree) {
  commandTree.listen('/tab/switch', ({
    argvNoOptions
  }) => {
    const idx = parseInt(argvNoOptions[argvNoOptions.length - 1], 10);

    _core.eventBus.emit('/tab/switch/request', idx - 1);

    return true;
  }, {
    usage
  });
  /**
   * Create and initialize a new tab
   *
   */

  commandTree.listen('/tab/new', args => __awaiter(this, void 0, void 0, function* () {
    // handle conditional tab creation
    if (args.parsedOptions.if) {
      // conditional opening request
      const condition = yield args.REPL.qexec(args.parsedOptions.if);

      if (!condition) {
        return true;
      }
    }

    if (args.parsedOptions.ifnot) {
      // conditional opening request
      const condition = yield args.REPL.qexec(args.parsedOptions.ifnot);

      if (condition) {
        return true;
      }
    } // status stripe decorations


    const message = args.parsedOptions['status-stripe-message'] || (args.execOptions.data ? args.execOptions.data['status-stripe-message'] : undefined);
    const statusStripeDecoration = {
      type: args.parsedOptions['status-stripe-type'],
      message
    }; // this is our response to the user if the tab was created
    // successfully

    const ok = {
      content: args.parsedOptions.title ? strings('Created a new tab named X', args.parsedOptions.title) : strings('Created a new tab'),
      contentType: 'text/markdown'
    };
    const file = args.parsedOptions.snapshot || args.parsedOptions.s;

    if (file) {
      // caller wants to open a given snapshot by file in the new tab
      const filepath = (0, _core.expandHomeDir)(file);
      const snapshot = yield loadSnapshotBuffer(args.REPL, filepath);
      return new Promise(resolve => {
        _core.eventBus.emit('/tab/new/request', {
          statusStripeDecoration,
          snapshot,
          title: args.parsedOptions.title,
          background: args.parsedOptions.bg,
          onClose: args.parsedOptions.onClose
        });

        resolve(ok);
      });
    } else if (args.parsedOptions.cmdline) {
      // caller wants to invoke a given command line in the new tab
      return new Promise(resolve => {
        _core.eventBus.emit('/tab/new/request', {
          statusStripeDecoration,
          title: args.parsedOptions.title,
          background: args.parsedOptions.bg,
          cmdline: args.parsedOptions.cmdline,
          exec: args.parsedOptions.quiet ? 'qexec' : 'pexec',
          onClose: args.parsedOptions.onClose
        });

        resolve(ok);
      });
    } else {
      // default case: tab opens without invoking a command line
      _core.eventBus.emit('/tab/new/request', {
        statusStripeDecoration,
        title: args.parsedOptions.title,
        background: args.parsedOptions.bg,
        onClose: args.parsedOptions.onClose
      });

      return ok;
    }
  }), {
    usage: {
      optional: [{
        name: '--cmdline',
        alias: '-c',
        docs: 'Invoke a command in the new tab'
      }, {
        name: '--quiet',
        alias: '-q',
        boolean: true,
        docs: 'Execute the given command line quietly'
      }, {
        name: '--bg',
        alias: '-b',
        boolean: true,
        docs: 'Create, but do not switch to this tab'
      }, {
        name: '--snapshot',
        alias: '-s',
        docs: 'Snapshot file to display in the new tab'
      }, {
        name: '--status-stripe-type',
        docs: 'Desired status stripe coloration',
        allowed: ['default', 'blue']
      }, {
        name: '--status-stripe-message',
        docs: 'Desired status stripe message'
      }, {
        name: '--title',
        alias: '-t',
        docs: 'Title to display in the UI'
      }]
    },
    flags: {
      boolean: ['bg', 'b', 'quiet', 'q']
    }
  });
  commandTree.listen('/tab/close', ({
    tab,
    parsedOptions
  }) => {
    return closeTab(tab, parsedOptions.A);
  }, {
    flags: {
      boolean: ['A']
    }
  });
}