"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.preload = preload;
exports.default = exports.NotebookVFS = void 0;

var _trieSearch = _interopRequireDefault(require("trie-search"));

var _micromatch = _interopRequireDefault(require("micromatch"));

var _posix = require("./posix");

var _fs = require("@kui-shell/plugin-bash-like/fs");

var _core = require("@kui-shell/core");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};
/* eslint-disable @typescript-eslint/no-unused-vars */


function isLeaf(entry) {
  return entry.data !== undefined;
}

const uid = -1;
const gid = -1;
const username = '';

class NotebookVFS {
  // eslint-disable-next-line no-useless-constructor
  constructor(mountPath = '/kui') {
    this.mountPath = mountPath;
    this.isLocal = false;
    this.isVirtual = true;
    this.prefix = new RegExp(`^${this.mountPath}\\/?`);
    this.trie = new _trieSearch.default();
  }
  /** Turn an ls-style glob into a nodejs-style regexp */


  glob2RegExp(filepath) {
    return filepath.replace(/\//g, '\\/').replace(/\*/g, '.*');
  }

  dirPattern(filepath) {
    // /kui/kubernetes -> /\/kui\/kubernetes/[^/]*$/
    if (filepath.charAt(filepath.length - 1) === '/') {
      return new RegExp(`^${this.glob2RegExp(filepath)}[^/]+$`);
    } else {
      return this.dirPattern(filepath + '/');
    }
  }
  /** Looks in the trie for any matches for the given filepath, handling the "contents of directory" case */


  find(filepath, dashD = false, exact = false) {
    const dirPattern = this.dirPattern(filepath);
    const flexMatches = this.trie.get(filepath.replace(/\*.*$/, '')).filter(_ => exact ? _.mountPath === filepath : _micromatch.default.isMatch(_.mountPath, filepath) || dirPattern.test(_.mountPath));

    if (dashD) {
      return flexMatches.filter(_ => _.isDirectory);
    } else if (exact) {
      return flexMatches;
    } else {
      return flexMatches.filter(_ => _.mountPath !== filepath);
    }
  }
  /** Looks in the trie for a single precise match */


  findExact(filepath, withData) {
    const possibleMatches = this.find(filepath, false, true);

    if (possibleMatches.length > 1) {
      const msg = 'Multiple matches';
      console.error(msg, possibleMatches);
      throw new Error(msg);
    } else if (possibleMatches.length === 0) {
      const flexMatches = this.find(filepath, false, false);

      if (filepath === this.mountPath || flexMatches.find(_ => _.mountPath.startsWith(filepath))) {
        // then this is either a match against the mount position or an interior directory
        return {
          viewer: 'ls',
          filepath,
          fullpath: filepath,
          size: 0,
          isDirectory: true
        };
      }
    } else {
      const entry = possibleMatches[0];
      return {
        viewer: 'replay',
        filepath: entry.mountPath,
        fullpath: entry.mountPath,
        size: isLeaf(entry) ? /\.json$/.test(entry.mountPath) ? JSON.stringify(entry.data, undefined, 2).length : entry.data.toString().length : 0,
        isDirectory: !isLeaf(entry),
        data: withData && isLeaf(entry) ? /\.json$/.test(entry.mountPath) ? JSON.stringify(entry.data, undefined, 2) : entry.data.toString() : undefined
      };
    }
  }

  enumerate({
    entries
  }) {
    return entries.map(mount => {
      const name = (0, _posix.basename)(mount.mountPath);
      const nameForDisplay = isLeaf(mount) && mount.data.metadata ? mount.data.metadata.description || mount.data.metadata.name || name : name;
      const isDir = !isLeaf(mount);
      return {
        name,
        nameForDisplay,
        path: mount.mountPath,
        stats: {
          size: 0,
          mtimeMs: 0,
          mode: 0,
          uid,
          gid
        },
        dirent: {
          mount: {
            isLocal: this.isLocal
          },
          isFile: !isDir,
          isDirectory: isDir,
          isSymbolicLink: false,
          isSpecial: false,
          isExecutable: false,
          permissions: '',
          username
        }
      };
    });
  }

  ls({
    parsedOptions
  }, filepaths) {
    return __awaiter(this, void 0, void 0, function* () {
      return (0, _core.flatten)(filepaths.map(filepath => ({
        filepath,
        entries: this.find(filepath, parsedOptions.d)
      })).filter(_ => _.entries.length > 0).map(_ => this.enumerate(_)));
    });
  }
  /** Insert filepath into directory */


  cp(_, srcFilepaths, dstFilepath) {
    return Promise.all(srcFilepaths.map(srcFilepath => {
      const match1 = srcFilepath.match(/^plugin:\/\/plugin-(.*)\/notebooks\/(.*)\.json$/);
      const match2 = srcFilepath.match(/^plugin:\/\/client\/notebooks\/(.*)\.json$/);
      const match3 = srcFilepath.match(/^plugin:\/\/client\/(.*)\.md$/);
      const match = match1 || match2 || match3;

      if (match) {
        try {
          // require versus import to work with babelized headless
          const file = match1 ? match1[2] : match2 ? match2[1] : match3[1];
          const data = match1 ? require('@kui-shell/plugin-' + match1[1] + '/notebooks/' + file + '.json') : match2 ? require('@kui-shell/client/notebooks/' + file + '.json') : require('@kui-shell/client/' + file + '.md').default;
          const extension = match1 || match2 ? '.json' : '.md';
          const dir = (0, _posix.dirname)(dstFilepath);

          if (!this.trie.get(dir)) {
            throw new Error(`Directory does not exist: ${dir}`);
          } else {
            const mountPath = (0, _posix.join)(dstFilepath, file + extension);
            this.trie.map(mountPath, {
              mountPath,
              data
            });
          }

          return;
        } catch (err) {
          console.error(err);
          throw new Error(`Unable to copy given source into the notebooks VFS: ${srcFilepath}. ${err.message}`);
        }
      } else {
        throw new Error(`Unable to copy given source into the notebooks VFS: ${srcFilepath}`);
      }
    })).then(() => 'ok');
  }
  /** Remove filepath */


  rm() {
    throw new Error('Unsupported operation');
  }
  /** Fetch contents */


  fstat(opts, filepath, withData, enoentOk) {
    return __awaiter(this, void 0, void 0, function* () {
      const entry = this.findExact(filepath, withData);

      if (!entry) {
        if (enoentOk) {// i don't think it makes sense to ignore ENOENT for this VFS
        }

        const error = new Error(`File not found: ${filepath}`);
        error.code = 404;
        throw error;
      } else {
        return entry;
      }
    });
  }

  fwrite(opts, filepath, data) {
    return __awaiter(this, void 0, void 0, function* () {
      throw new Error('Unsupported operation');
    });
  }
  /** Fetch content slice */


  fslice(filename, offset, length) {
    return __awaiter(this, void 0, void 0, function* () {
      const entry = this.findExact(filename, true);

      if (entry.data) {
        const buffer = Buffer.from(entry.data);

        if (offset > buffer.length) {
          throw new Error(`notebook fslice: reach file end`);
        } else {
          return buffer.slice(offset, length + offset).toString();
        }
      } else {
        throw new Error(`fslice: data not found ${filename}`);
      }
    });
  }
  /** Create a directory/bucket */


  mkdir(opts) {
    return __awaiter(this, void 0, void 0, function* () {
      const mountPath = opts.argvNoOptions[opts.argvNoOptions.indexOf('mkdir') + 1];
      this.trie.map(mountPath, {
        mountPath,
        isDirectory: true
      });
    });
  }
  /** Remove a directory/bucket */


  rmdir() {
    throw new Error('Unsupported operation');
  }

}

exports.NotebookVFS = NotebookVFS;
const vfs = new NotebookVFS();
var _default = vfs;
exports.default = _default;

function preload() {
  (0, _fs.mount)(vfs);
}