/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { Common, CLI, ReplExpect, Selectors, Util } from '@kui-shell/test';
const outputs = ['smurf', 'dragon'];
const inputs = outputs.map(_ => Buffer.from(_).toString('base64'));
import { doSplitViaButton } from '../core-support2/split-helpers';
import { tabButtonSelector } from '../../lib/cmds/tab-management';
describe('block copy paste command', function () {
    before(Common.before(this));
    after(Common.after(this));
    Util.closeAllExceptFirstTab.bind(this)();
    const exec = (idx = 0) => __awaiter(this, void 0, void 0, function* () {
        const res = yield CLI.command(`base64 --decode ${inputs[idx]}`, this.app);
        yield ReplExpect.okWithString(outputs[idx])(res);
        return res.count;
    });
    const focusBlock = (N) => __awaiter(this, void 0, void 0, function* () {
        yield this.app.client.$(Selectors.PROMPT_CONTEXT_N(N)).then(_ => _.click());
    });
    const execAndCopy = () => __awaiter(this, void 0, void 0, function* () {
        const N = yield exec();
        yield focusBlock(N);
        yield this.app.client.waitUntil(() => this.app.client.$(Selectors.PROMPT_BLOCK_N(N)).then(_ => _.isFocused()), {
            timeout: CLI.waitTimeout
        });
        yield this.app.client.execute(() => document.execCommand('copy'));
        return N;
    });
    /**
     * @param N the block index to cut
     *
     */
    const cut = (N) => __awaiter(this, void 0, void 0, function* () {
        yield focusBlock(N);
        yield this.app.client.execute(() => document.execCommand('cut'));
    });
    /**
     * @param N the block index in which we expect to find the pasted block
     * @param idx index into the `outputs[]` variable; i.e. a pointer to the expected string
     * @param splitIndex the split index in which we expect to find the pasted block
     *
     */
    const pasteAndVerify = (N, idx = 0, splitIndex = 1) => __awaiter(this, void 0, void 0, function* () {
        yield this.app.client.execute(() => document.execCommand('paste'));
        yield ReplExpect.okWithString(outputs[idx])({ app: this.app, count: N, splitIndex });
    });
    it('block copy by keeping the focus on the prior context', () => __awaiter(this, void 0, void 0, function* () {
        try {
            const N = yield execAndCopy();
            yield pasteAndVerify(N + 1);
        }
        catch (err) {
            yield Common.oops(this, true)(err);
        }
    }));
    it('should execute an intervening new base64', () => exec(1).catch(Common.oops(this, true)));
    const expectBlockCount = ReplExpect.blockCount.bind(this);
    it('block copy by focusing on the next active prompt, then cut', () => __awaiter(this, void 0, void 0, function* () {
        try {
            const N = yield execAndCopy();
            yield CLI.grabFocus(this.app); // focus on the next active prompt
            yield pasteAndVerify(N + 1);
            yield cut(N + 1);
            yield expectBlockCount()
                .inSplit(1)
                .is(N + 1);
            yield cut(N);
            yield expectBlockCount()
                .inSplit(1)
                .is(N);
            yield cut(N - 1);
            yield expectBlockCount()
                .inSplit(1)
                .is(N - 1);
        }
        catch (err) {
            yield Common.oops(this, true)(err);
        }
    }));
    it('click new tab button', () => this.app.client
        .$(tabButtonSelector)
        .then(_ => _.click())
        .then(() => this.app.client.$(Selectors.TAB_SELECTED_N(2)))
        .then(_ => _.waitForDisplayed())
        .then(() => CLI.waitForRepl(this.app)) // should have an active repl
        .catch(Common.oops(this, true)));
    it('should paste the block in the new tab', () => __awaiter(this, void 0, void 0, function* () {
        try {
            const N = yield CLI.nOfCurrentBlock(this.app);
            yield pasteAndVerify(N, 1); // re: 1, since we cut until we reached the "intervening new base64"
        }
        catch (err) {
            yield Common.oops(this, true)(err);
        }
    }));
    it('should create a new split via button click and paste the block in that new split', () => __awaiter(this, void 0, void 0, function* () {
        try {
            const splitIndex = 2;
            yield doSplitViaButton(this, splitIndex);
            const N = yield CLI.nOfCurrentBlock(this.app, splitIndex);
            yield pasteAndVerify(N, 1, splitIndex); // re: 1, same as above!
        }
        catch (err) {
            yield Common.oops(this, true)(err);
        }
    }));
});
//# sourceMappingURL=copy-block.js.map