"use strict";

var _react = _interopRequireDefault(require("react"));

var _Tile = require("../Tile");

var _enzyme = require("enzyme");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Copyright IBM Corp. 2016, 2018
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */
var prefix = 'bx';
describe('Tile', function () {
  describe('Renders default tile as expected', function () {
    var wrapper = (0, _enzyme.shallow)( /*#__PURE__*/_react.default.createElement(_Tile.Tile, {
      className: "extra-class"
    }, /*#__PURE__*/_react.default.createElement("div", {
      className: "child"
    }, "Test")));
    it('renders children as expected', function () {
      expect(wrapper.find('.child').length).toBe(1);
    });
    it('renders extra classes passed in via className', function () {
      expect(wrapper.hasClass('extra-class')).toEqual(true);
    });
    it('supports light version', function () {
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_Tile.Tile, null, "Test"));
      expect(wrapper.props().light).toEqual(false);
      expect(wrapper.childAt(0).hasClass("".concat(prefix, "--tile--light"))).toEqual(false);
      wrapper.setProps({
        light: true
      });
      expect(wrapper.props().light).toEqual(true);
      expect(wrapper.childAt(0).hasClass("".concat(prefix, "--tile--light"))).toEqual(true);
    });
  });
  describe('Renders clickable tile as expected', function () {
    var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_Tile.ClickableTile, {
      className: "extra-class"
    }, /*#__PURE__*/_react.default.createElement("div", {
      className: "child"
    }, "Test")));
    beforeEach(function () {
      wrapper.state().clicked = false;
    });
    it('renders children as expected', function () {
      expect(wrapper.find('.child').length).toBe(1);
    });
    it('renders extra classes passed in via className', function () {
      expect(wrapper.hasClass('extra-class')).toEqual(true);
    });
    it('toggles the clickable class on click', function () {
      expect(wrapper.find('Link').hasClass("".concat(prefix, "--tile--is-clicked"))).toEqual(false);
      wrapper.simulate('click', {
        persist: function persist() {}
      });
      expect(wrapper.find('Link').hasClass("".concat(prefix, "--tile--is-clicked"))).toEqual(true);
    });
    it('toggles the clickable state on click', function () {
      expect(wrapper.state().clicked).toEqual(false);
      wrapper.simulate('click', {
        persist: function persist() {}
      });
      expect(wrapper.state().clicked).toEqual(true);
    });
    it('toggles the clicked state when using enter or space', function () {
      expect(wrapper.state().clicked).toEqual(false);
      wrapper.simulate('keydown', {
        which: 32,
        persist: function persist() {}
      });
      expect(wrapper.state().clicked).toEqual(true);
      wrapper.simulate('keydown', {
        which: 13,
        persist: function persist() {}
      });
      expect(wrapper.state().clicked).toEqual(false);
    });
    it('supports setting initial clicked state from props', function () {
      expect((0, _enzyme.shallow)( /*#__PURE__*/_react.default.createElement(_Tile.ClickableTile, {
        clicked: true
      })).state().clicked).toEqual(true);
    });
    it('supports setting clicked state from props', function () {
      wrapper.setProps({
        clicked: true
      });
      wrapper.setState({
        clicked: true
      });
      wrapper.setProps({
        clicked: false
      });
      expect(wrapper.state().clicked).toEqual(false);
    });
    it('avoids changing clicked state upon setting props, unless actual value change is detected', function () {
      wrapper.setProps({
        clicked: true
      });
      wrapper.setState({
        clicked: false
      });
      wrapper.setProps({
        clicked: true
      });
      expect(wrapper.state().clicked).toEqual(false);
    });
    it('supports light version', function () {
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_Tile.ClickableTile, null, "Test"));
      expect(wrapper.props().light).toEqual(false);
      expect(wrapper.childAt(0).hasClass("".concat(prefix, "--tile--light"))).toEqual(false);
      wrapper.setProps({
        light: true
      });
      expect(wrapper.props().light).toEqual(true);
      expect(wrapper.childAt(0).hasClass("".concat(prefix, "--tile--light"))).toEqual(true);
    });
  });
  describe('Renders selectable tile as expected', function () {
    var wrapper;
    var label;
    beforeEach(function () {
      wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_Tile.SelectableTile, {
        className: "extra-class",
        onClick: jest.fn()
      }, /*#__PURE__*/_react.default.createElement("div", {
        className: "child"
      }, "Test")));
      label = wrapper.find('label');
    });
    it('renders children as expected', function () {
      expect(wrapper.find('.child').length).toBe(1);
    });
    it('renders extra classes passed in via className', function () {
      expect(wrapper.hasClass('extra-class')).toEqual(true);
    });
    it('toggles the selectable state on click', function () {
      expect(wrapper.hasClass("".concat(prefix, "--tile--is-selected"))).toEqual(false);
      label.simulate('click');
      expect(wrapper.props().onClick).toHaveBeenCalledTimes(1);
      expect(wrapper.render().hasClass("".concat(prefix, "--tile--is-selected"))).toEqual(true);
    });
    it('toggles the selectable state when using enter or space', function () {
      expect(wrapper.hasClass("".concat(prefix, "--tile--is-selected"))).toEqual(false);
      label.simulate('keydown', {
        which: 32
      });
      expect(wrapper.render().hasClass("".concat(prefix, "--tile--is-selected"))).toEqual(true);
      label.simulate('keydown', {
        which: 13
      });
      expect(wrapper.render().hasClass("".concat(prefix, "--tile--is-selected"))).toEqual(false);
    });
    it('the input should be checked when state is selected', function () {
      label.simulate('click');
      expect(wrapper.find('input').props().checked).toEqual(true);
    });
    it('supports setting initial selected state from props', function () {
      expect((0, _enzyme.shallow)( /*#__PURE__*/_react.default.createElement(_Tile.SelectableTile, {
        selected: true
      })).render().hasClass("".concat(prefix, "--tile--is-selected"))).toEqual(true);
    });
    it('supports setting selected state from props', function () {
      wrapper.setProps({
        selected: true
      });
      expect(wrapper.render().hasClass("".concat(prefix, "--tile--is-selected"))).toEqual(true);
    });
    it('avoids changing selected state upon setting props, unless actual value change is detected', function () {
      wrapper.setProps({
        selected: true
      });
      label.simulate('click');
      wrapper.setProps({
        selected: true
      });
      expect(wrapper.hasClass("".concat(prefix, "--tile--is-selected"))).toEqual(false);
    });
    it('supports light version', function () {
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_Tile.SelectableTile, null, "Test"));
      expect(wrapper.props().light).toEqual(false);
      expect(wrapper.childAt(1).hasClass('bx--tile--light')).toEqual(false);
      wrapper.setProps({
        light: true
      });
      expect(wrapper.props().light).toEqual(true);
      expect(wrapper.childAt(1).hasClass('bx--tile--light')).toEqual(true);
    });
    it('should call onChange when the checkbox value changes', function () {
      var onChange = jest.fn();
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_Tile.SelectableTile, {
        onChange: onChange
      }, /*#__PURE__*/_react.default.createElement("span", {
        id: "test-id"
      }, "test")));
      var content = wrapper.find('#test-id'); // Tile becomes selected

      content.simulate('click');
      expect(onChange).toHaveBeenCalledTimes(1); // Tile becomes un-selected

      content.simulate('click');
      expect(onChange).toHaveBeenCalledTimes(2);
    });
    it('supports disabled state', function () {
      wrapper.setProps({
        disabled: true
      });
      expect(wrapper.find('input').props().disabled).toEqual(true);
    });
  });
  describe('Renders expandable tile as expected', function () {
    var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_Tile.ExpandableTile, {
      className: "extra-class"
    }, /*#__PURE__*/_react.default.createElement(_Tile.TileAboveTheFoldContent, {
      className: "child"
    }, /*#__PURE__*/_react.default.createElement("div", {
      style: {
        height: '200px'
      }
    }, "Test")), /*#__PURE__*/_react.default.createElement(_Tile.TileBelowTheFoldContent, {
      className: "child"
    }, /*#__PURE__*/_react.default.createElement("div", {
      style: {
        height: '500px'
      }
    }, "Test"), /*#__PURE__*/_react.default.createElement("a", {
      id: "test-link",
      href: "/"
    }, "Test Link"))));
    beforeEach(function () {
      wrapper.state().expanded = false;
    });
    it('renders children as expected', function () {
      expect(wrapper.props().children.length).toBe(2);
    });
    it('has the expected classes', function () {
      expect(wrapper.children().hasClass("".concat(prefix, "--tile--expandable"))).toEqual(true);
    });
    it('renders extra classes passed in via className', function () {
      expect(wrapper.hasClass('extra-class')).toEqual(true);
    });
    it('toggles the expandable class on click', function () {
      expect(wrapper.children().hasClass("".concat(prefix, "--tile--is-expanded"))).toEqual(false);
      wrapper.simulate('click');
      expect(wrapper.children().hasClass("".concat(prefix, "--tile--is-expanded"))).toEqual(true);
    });
    it('toggles the expandable state on click', function () {
      expect(wrapper.state().expanded).toEqual(false);
      wrapper.simulate('click');
      expect(wrapper.state().expanded).toEqual(true);
    });
    it('ignores allows click events to be ignored using onBeforeClick', function () {
      wrapper.setProps({
        onBeforeClick: function onBeforeClick(evt) {
          return evt.target.tagName.toLowerCase() !== 'a';
        } // ignore link clicks

      });
      expect(wrapper.state().expanded).toEqual(false);
      wrapper.simulate('click');
      expect(wrapper.state().expanded).toEqual(true);
      wrapper.find('#test-link').simulate('click');
      expect(wrapper.state().expanded).toEqual(true);
      wrapper.simulate('click');
      expect(wrapper.state().expanded).toEqual(false);
    });
    it('displays the default tooltip for the button depending on state', function () {
      var defaultExpandedIconText = 'Interact to collapse Tile';
      var defaultCollapsedIconText = 'Interact to expand Tile'; // Force the expanded tile to be collapsed.

      wrapper.setState({
        expanded: false
      });
      var collapsedDescription = wrapper.find('button').prop('title');
      expect(collapsedDescription).toEqual(defaultCollapsedIconText); // click on the item to expand it.

      wrapper.simulate('click'); // Validate the description change

      var expandedDescription = wrapper.find('button').prop('title');
      expect(expandedDescription).toEqual(defaultExpandedIconText);
    });
    it('displays the custom tooltips for the button depending on state', function () {
      var tileExpandedIconText = 'Click To Collapse';
      var tileCollapsedIconText = 'Click To Expand'; // Force the custom icon text

      wrapper.setProps({
        tileExpandedIconText: tileExpandedIconText,
        tileCollapsedIconText: tileCollapsedIconText
      }); // Force the expanded tile to be collapsed.

      wrapper.setState({
        expanded: false
      });
      var collapsedDescription = wrapper.find('button').prop('title');
      expect(collapsedDescription).toEqual(tileCollapsedIconText); // click on the item to expand it.

      wrapper.simulate('click'); // Validate the description change

      var expandedDescription = wrapper.find('button').prop('title');
      expect(expandedDescription).toEqual(tileExpandedIconText);
    });
    it('supports setting initial expanded state from props', function () {
      var _mount$state = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_Tile.ExpandableTile, {
        expanded: true
      }, /*#__PURE__*/_react.default.createElement(_Tile.TileAboveTheFoldContent, {
        className: "child"
      }, /*#__PURE__*/_react.default.createElement("div", {
        style: {
          height: '200px'
        }
      }, "Test")), /*#__PURE__*/_react.default.createElement(_Tile.TileBelowTheFoldContent, {
        className: "child"
      }, /*#__PURE__*/_react.default.createElement("div", {
        style: {
          height: '500px'
        }
      }, "Test")))).state(),
          expanded = _mount$state.expanded;

      expect(expanded).toEqual(true);
    });
    it('supports setting expanded state from props', function () {
      wrapper.setProps({
        expanded: true
      });
      wrapper.setState({
        expanded: true
      });
      wrapper.setProps({
        expanded: false
      });
      expect(wrapper.state().expanded).toEqual(false);
    });
    it('avoids changing expanded state upon setting props, unless actual value change is detected', function () {
      wrapper.setProps({
        expanded: true
      });
      wrapper.setState({
        expanded: false
      });
      wrapper.setProps({
        expanded: true
      });
      expect(wrapper.state().expanded).toEqual(false);
    });
    it('supports setting max height from props', function () {
      wrapper.setProps({
        tileMaxHeight: 2
      });
      wrapper.setState({
        tileMaxHeight: 2
      });
      wrapper.setProps({
        tileMaxHeight: 1
      });
      expect(wrapper.state().tileMaxHeight).toEqual(1);
    });
    it('avoids changing max height upon setting props, unless actual value change is detected', function () {
      wrapper.setProps({
        tileMaxHeight: 2
      });
      wrapper.setState({
        tileMaxHeight: 1
      });
      wrapper.setProps({
        tileMaxHeight: 2
      });
      expect(wrapper.state().tileMaxHeight).toEqual(1);
    });
    it('supports setting padding from props', function () {
      wrapper.setProps({
        tilePadding: 2
      });
      wrapper.setState({
        tilePadding: 2
      });
      wrapper.setProps({
        tilePadding: 1
      });
      expect(wrapper.state().tilePadding).toEqual(1);
    });
    it('avoids changing padding upon setting props, unless actual value change is detected', function () {
      wrapper.setProps({
        tilePadding: 2
      });
      wrapper.setState({
        tilePadding: 1
      });
      wrapper.setProps({
        tilePadding: 2
      });
      expect(wrapper.state().tilePadding).toEqual(1);
    });
    it('supports light version', function () {
      var wrapper = (0, _enzyme.mount)( /*#__PURE__*/_react.default.createElement(_Tile.ExpandableTile, null, "Test"));
      expect(wrapper.props().light).toEqual(false);
      expect(wrapper.childAt(0).hasClass("".concat(prefix, "--tile--light"))).toEqual(false);
      wrapper.setProps({
        light: true
      });
      expect(wrapper.props().light).toEqual(true);
      expect(wrapper.childAt(0).hasClass("".concat(prefix, "--tile--light"))).toEqual(true);
    });
  });
});