"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTable = getTable;
exports.get = get;
exports.default = getDirect;

var _core = require("@kui-shell/core");

var _watch = _interopRequireDefault(require("./watch"));

var _fetchFile = require("../../../lib/util/fetch-file");

var _util = require("../../../lib/util/util");

var _formatTable = require("../../../lib/view/formatTable");

var _status = require("../../kubectl/status");

var _options = require("../../kubectl/options");

var _columns = _interopRequireDefault(require("./columns"));

var _url = require("./url");

var _errors = _interopRequireWildcard(require("./errors"));

var _headers = require("./headers");

var _resource = require("../../../lib/model/resource");

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = void 0 && (void 0).__awaiter || function (thisArg, _arguments, P, generator) {
  function adopt(value) {
    return value instanceof P ? value : new P(function (resolve) {
      resolve(value);
    });
  }

  return new (P || (P = Promise))(function (resolve, reject) {
    function fulfilled(value) {
      try {
        step(generator.next(value));
      } catch (e) {
        reject(e);
      }
    }

    function rejected(value) {
      try {
        step(generator["throw"](value));
      } catch (e) {
        reject(e);
      }
    }

    function step(result) {
      result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected);
    }

    step((generator = generator.apply(thisArg, _arguments || [])).next());
  });
};

function getTable(drilldownCommand, namespace, names, explainedKind, format, args, needsStatusColumn = false, customColumns) {
  return __awaiter(this, void 0, void 0, function* () {
    const {
      kind
    } = explainedKind;
    const group = {
      explainedKind,
      names,
      namespace
    };
    const formatUrl = yield (0, _url.urlFormatterFor)(drilldownCommand, namespace, args, explainedKind);
    const urls = names.length === 0 ? formatUrl(true, true) : names.map(formatUrl.bind(undefined, true, true)).join(',');
    const fmt = format || 'default';
    const isCusto = (0, _options.isCustomColumns)(fmt);

    if (fmt === 'wide' || fmt === 'default' || isCusto) {
      // first, fetch the data; we pass returnErrors=true here, so that we can assemble 404s properly
      const responses = yield (0, _fetchFile.fetchFile)(args.REPL, urls, {
        headers: isCusto ? _headers.headersForPlainRequest : _headers.headersForTableRequest,
        returnErrors: true
      }); // then dissect it into errors and non-errors

      const {
        errors,
        ok
      } = yield (0, _errors.default)(responses, formatUrl, kind, args.REPL);

      if (isCusto) {
        const list = ok.reduce((list, data) => {
          const thisList = Buffer.isBuffer(data) || typeof data === 'string' ? JSON.parse(data.toString()) : data;

          if (!list) {
            // first table response
            return thisList;
          } else {
            // accumulate list responses
            list.items = list.items.concat(thisList.items);
            return list;
          }
        }, undefined);
        list.isKubeResource = true;
        const table = (yield Promise.resolve().then(() => require('./custom-columns'))).toKuiTableFromCustomColumns(list, args, drilldownCommand, kind, fmt);
        return !(0, _options.isWatchRequest)(args) ? table : (0, _watch.default)(drilldownCommand, args, kind, group, table, formatUrl);
      } else {
        // assemble the non-errors into a single table
        const metaTable = ok.reduce((metaTable, data) => {
          const thisTable = Buffer.isBuffer(data) || typeof data === 'string' ? JSON.parse(data.toString()) : data;

          if (!metaTable) {
            // first table response
            return thisTable;
          } else {
            // accumulate table responses
            metaTable.rows = metaTable.rows.concat(thisTable.rows);
            return metaTable;
          }
        }, undefined);

        if (args.execOptions.type === _core.ExecType.TopLevel && metaTable && (!metaTable.rows || metaTable.rows.length === 0) && !(0, _options.isWatchRequest)(args)) {
          // throw error so UI can render an error block for this response
          const err = new Error(`No resources found in **${namespace}** namespace.`);
          err.code = 404;
          throw err;
        } else {
          try {
            // withNotFound will add error rows to the table for each error
            const table = (0, _formatTable.withNotFound)(yield (0, _formatTable.toKuiTable)(metaTable, kind, args, drilldownCommand, needsStatusColumn, customColumns), errors.map(_ => _.message).join('\n'));
            return !(0, _options.isWatchRequest)(args) ? table : (0, _watch.default)(drilldownCommand, args, kind, group, table, formatUrl);
          } catch (err) {
            console.error('error formatting table', err);
            throw new Error('Internal Error');
          }
        }
      }
    }
  });
}
/**
 * Direct `get` only handles the following cases:
 *
 * 1. file request with table output, e.g. `kubectl get -f` and `kubectl get -k`
 * 2. table request, e.g. `kubectl get pods` and `kubectl get pod nginx`
 * 3. entity request with kind and name, e.g. `get pod nginx -o yaml`
 *
 * TODO 1: consolidate `1` and `2` into a single table request handler using `doStatus`
 * TODO 2: handle entity request with file
 *
 */


function get(drilldownCommand, namespace, names, explainedKind, format, args) {
  return __awaiter(this, void 0, void 0, function* () {
    const isFileRequest = (0, _options.getFileFromArgv)(args);
    const isEntityAndNameFormat = (0, _options.isEntityFormat)(format) || format === 'name';
    /** 1. file request with table output, e.g. `kubectl get -f` and `kubectl get -k` */

    if (isFileRequest && !isEntityAndNameFormat) {
      return (0, _status.doStatus)(args, 'get', drilldownCommand, undefined, undefined, undefined, (0, _options.isWatchRequest)(args));
    }
    /** 2. table request, e.g. `kubectl get pods` and `kubectl get pod nginx` */


    if ((0, _options.isTableRequest)(args)) {
      return getTable(drilldownCommand, namespace, names, explainedKind, format, args, undefined, (0, _columns.default)(explainedKind.kind, args));
    }
    /** 3. entity request with kind and name, e.g. `get pod nginx -o yaml` */


    if (!(0, _options.isTableRequest)(args) && !isFileRequest && args.parsedOptions.kubeconfig === undefined && args.parsedOptions.context === undefined && isEntityAndNameFormat) {
      const formatUrl = yield (0, _url.urlFormatterFor)(drilldownCommand, namespace, args, explainedKind);
      const urls = names.length === 0 ? formatUrl(true, true) : names.map(formatUrl.bind(undefined, true, true)).join(',');
      let response;

      try {
        response = (yield (0, _fetchFile.fetchFile)(args.REPL, urls, {
          headers: {
            accept: 'application/json'
          }
        }))[0];
      } catch (err) {
        response = (0, _errors.tryParseAsStatus)(err.code, err.message);

        if (!(0, _resource.isStatus)(response)) {
          throw err;
        }
      }

      if ((0, _resource.isStatus)(response)) {
        const error = new Error(`Error from server (${response.reason}): ${response.message}`);
        error.code = response.code;
        throw error;
      } else if (format === 'name') {
        try {
          const stdout = (Buffer.isBuffer(response) || typeof response === 'string' ? JSON.parse(response.toString()) : response).items.map(_ => _.metadata.name).join('\n');
          return {
            content: {
              code: 0,
              stderr: '',
              stdout,
              wasSentToPty: false
            }
          };
        } catch (err) {
          throw new Error(response.toString());
        }
      } else if (format === 'yaml') {
        const {
          dump
        } = yield Promise.resolve().then(() => require('js-yaml'));
        return {
          content: {
            code: 0,
            stderr: '',
            stdout: dump(Buffer.isBuffer(response) || typeof response === 'string' ? JSON.parse(response.toString()) : response),
            wasSentToPty: false
          }
        };
      } else {
        return {
          content: {
            code: 0,
            stderr: '',
            stdout: Buffer.isBuffer(response) || typeof response === 'string' ? response.toString() : JSON.stringify(response),
            wasSentToPty: false
          }
        };
      }
    }
  });
}

function getDirect(args, _kind) {
  return __awaiter(this, void 0, void 0, function* () {
    const namespace = (0, _options.getNamespace)(args);
    const format = (0, _options.formatOf)(args);
    const drilldownCommand = (0, _util.getCommandFromArgs)(args);
    const kindIdx = args.argvNoOptions.indexOf('get') + 1;
    const names = args.argvNoOptions.slice(kindIdx + 1);
    const explainedKind = _kind ? yield _kind : {
      kind: undefined,
      version: undefined,
      isClusterScoped: false
    };
    return get(drilldownCommand, yield namespace, names, explainedKind, format, args);
  });
}