"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.cpuFraction = cpuFraction;
exports.cpuShare = cpuShare;
exports.memShare = memShare;
exports.formatAsBytes = formatAsBytes;
exports.reformatAsBytes = reformatAsBytes;
exports.fractionOfMemory = fractionOfMemory;
exports.formatAsCpu = formatAsCpu;
exports.parseAsTime = parseAsTime;
exports.fromSize = fromSize;
exports.sumTime = sumTime;
exports.sumSize = sumSize;
exports.parseAsSize = parseAsSize;
exports.default = void 0;

var _bytesIec = _interopRequireDefault(require("bytes-iec"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2019 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
function cpuFraction(str) {
  return parseInt(str.replace(/%$/, ''), 10);
}

function cpuShare(str) {
  if (/m$/.test(str)) {
    return parseInt(str.replace(/m$/, ''), 10);
  } else {
    return parseInt(str, 10) * 1000;
  }
}

const byte = 1;
const kilobytes = 1024 * byte;
const megabytes = 1024 * kilobytes;
const gigabytes = 1024 * megabytes;
const terabytes = 1024 * gigabytes;
const petabytes = 1024 * terabytes;
const exabytes = 1024 * petabytes;

function memShare(str) {
  let end = 2;
  let unit = byte;

  if (str === '0') {
    return 0;
  } else if (/Ei$/.test(str)) {
    unit = exabytes;
  } else if (/Pi$/.test(str)) {
    unit = petabytes;
  } else if (/Ti$/.test(str)) {
    unit = terabytes;
  } else if (/Gi$/.test(str)) {
    unit = gigabytes;
  } else if (/Mi$/.test(str)) {
    unit = megabytes;
  } else if (/Ki$/.test(str)) {
    unit = kilobytes;
  } else if (/\d+/.test(str)) {
    return parseInt(str);
  } else {
    end = 1;
  }

  return parseInt(str.slice(0, str.length - end), 10) * unit;
}

function formatAsBytes(mem) {
  if (mem < kilobytes) {
    return mem.toString();
  } else if (mem < 10 * megabytes) {
    return (mem / kilobytes).toFixed(0) + 'Ki';
  } else if (mem < 10 * gigabytes) {
    return (mem / megabytes).toFixed(0) + 'Mi';
  } else if (mem < 10 * terabytes) {
    return (mem / gigabytes).toFixed(0) + 'Gi';
  } else if (mem < 10 * petabytes) {
    return (mem / terabytes).toFixed(0) + 'Ti';
  }
  /* if (mem < 10 * petabytes) */
  else {
      return (mem / petabytes).toFixed(0) + 'Pi';
    }
}
/** Turn e.g. 4041544Ki into 3947Mi */


function reformatAsBytes(mem) {
  return formatAsBytes(memShare(mem));
}
/** Fraction of two string-form memory figures */


function fractionOfMemory(num, denom) {
  return (memShare(num) / memShare(denom) * 100).toFixed(1) + '%';
}

function formatAsCpu(cpu) {
  return cpu > 10000 ? (cpu / 1000).toFixed(0) : `${cpu}m`;
}

function parseAsTime(str) {
  return cpuShare(str).toString();
}

function fromSize(str) {
  return _bytesIec.default.parse(str.replace(/m/g, 'MB').replace(/Ki/g, 'KiB').replace(/Mi/g, 'MiB').replace(/Gi/g, 'GiB').replace(/Ti/g, 'TiB').replace(/Pi/g, 'PiB').replace(/Ei/g, 'EiB'));
}
/**
 * @return the sum of the attributes in the given column, interpreted
 * as times
 *
 */


function sumTime(table, attrIdx, backupAttrIdx = attrIdx) {
  return table.body.map(_ => cpuShare(_.attributes[attrIdx].value === '0' ? _.attributes[backupAttrIdx].value : _.attributes[attrIdx].value)).reduce((sum, _) => sum + _, 0);
}
/**
 * @return the sum of the attributes in the given column, interpreted as sizes
 *
 */


function sumSize(table, attrIdx, backupAttrIdx = attrIdx) {
  return table.body.map(_ => fromSize(_.attributes[attrIdx].value === '0Ki' ? _.attributes[backupAttrIdx].value : _.attributes[attrIdx].value)).reduce((sum, _) => sum + _, 0);
}

function parseAsSize(str) {
  return (0, _bytesIec.default)(fromSize(str), {});
}
/** Rollup export */


const Parser = {
  fractionOfMemory,
  reformatAsBytes,
  formatAsBytes,
  formatAsCpu,
  cpuShare,
  memShare,
  cpuFraction
};
var _default = Parser;
exports.default = _default;