/*
 * Copyright 2019 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { Common, CLI, ReplExpect, Selectors } from '@kui-shell/test';
import { waitForGreen, waitForTerminalText, createNS, allocateNS, deleteNS } from '@kui-shell/plugin-kubectl/tests/lib/k8s/utils';
import { readFileSync } from 'fs';
import { dirname, join } from 'path';
const ROOT = dirname(require.resolve('@kui-shell/plugin-kubectl/tests/package.json'));
const inputBuffer = readFileSync(join(ROOT, 'data/k8s/kubectl-exec.yaml'));
const inputEncoded = inputBuffer.toString('base64');
const wdescribe = !process.env.USE_WATCH_PANE ? describe : xdescribe;
wdescribe(`kubectl logs getty via table ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    before(Common.before(this));
    after(Common.after(this));
    const ns = createNS();
    const inputs = [
        {
            podName: 'vim',
            containerName: 'alpine',
            hasLogs: true,
            expectString: true,
            cmdline: `echo ${inputEncoded} | base64 --decode | kubectl create -f - -n ${ns}`
        },
        {
            podName: 'nginx',
            containerName: 'nginx',
            label: 'name=nginx',
            hasLogs: false,
            expectString: false,
            cmdline: `kubectl create -f https://raw.githubusercontent.com/kubernetes/examples/master/staging/pod -n ${ns}`
        }
    ];
    const createPodExpectingString = (podName, cmdline) => {
        it(`should create ${podName} pod expect string`, () => {
            return CLI.command(cmdline, this.app)
                .then(ReplExpect.okWithPtyOutput(podName))
                .catch(Common.oops(this, true));
        });
    };
    const createPodExpectingTable = (podName, cmdline) => {
        it(`should create ${podName} pod expect table`, () => {
            return CLI.command(cmdline, this.app)
                .then(ReplExpect.okWith(podName))
                .catch(Common.oops(this, true));
        });
    };
    const waitForPod = (podName) => {
        it(`should wait for the pod ${podName} to come up`, () => {
            return CLI.command(`kubectl get pod ${podName} -n ${ns} -w`, this.app)
                .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME(podName) }))
                .then(selector => waitForGreen(this.app, selector))
                .catch(Common.oops(this, true));
        });
    };
    const waitForLogText = waitForTerminalText.bind(this);
    let res;
    const showLogs = (podName, containerName, label, hasLogs) => {
        const checkLogs = (res) => __awaiter(this, void 0, void 0, function* () {
            yield ReplExpect.ok(res);
            if (hasLogs) {
                yield waitForLogText(res, 'hi');
            }
        });
        it(`should show logs for pod ${podName} container ${containerName}`, () => __awaiter(this, void 0, void 0, function* () {
            try {
                res = yield CLI.command(`kubectl logs ${podName} ${containerName} -n ${ns}`, this.app);
                yield checkLogs(res);
            }
            catch (err) {
                yield Common.oops(this, true)(err);
            }
        }));
        if (label) {
            it(`should show logs for label selector ${label}`, () => __awaiter(this, void 0, void 0, function* () {
                try {
                    res = yield CLI.command(`kubectl logs -l${label} -n ${ns}`, this.app);
                    yield checkLogs(res);
                }
                catch (err) {
                    yield Common.oops(this, true)(err);
                }
            }));
        }
    };
    const doRetry = (hasLogs) => {
        it('should click retry button', () => __awaiter(this, void 0, void 0, function* () {
            try {
                yield this.app.client.$(Selectors.SIDECAR_MODE_BUTTON(res.count, 'retry-streaming')).then((_) => __awaiter(this, void 0, void 0, function* () {
                    yield _.waitForDisplayed();
                    yield _.click();
                }));
                if (hasLogs) {
                    yield waitForLogText(res, 'hi');
                }
            }
            catch (err) {
                return Common.oops(this, true)(err);
            }
        }));
    };
    allocateNS(this, ns);
    // needed to force the dom renderer for webpack/browser-based tests; see ExecIntoPod
    Common.setDebugMode.bind(this)();
    inputs.forEach(_ => {
        if (_.expectString) {
            createPodExpectingString(_.podName, _.cmdline);
        }
        else {
            createPodExpectingTable(_.podName, _.cmdline);
        }
        waitForPod(_.podName);
        showLogs(_.podName, _.containerName, _.label, _.hasLogs);
        doRetry(_.hasLogs);
    });
    inputs.forEach(_ => {
        showLogs(_.podName, _.containerName, _.label, _.hasLogs);
    });
    deleteNS(this, ns);
});
//# sourceMappingURL=logs-via-table.js.map