/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { isCodedError } from '@kui-shell/core';
import { headersForPlainRequest } from './headers';
import { isStatus } from '../../../lib/model/resource';
import { fetchFile, isReturnedError } from '../../../lib/util/fetch-file';
/** See if the given error message is a Kubernetes Status object */
export function tryParseAsStatus(code, error) {
    try {
        const obj = JSON.parse(error);
        if (isStatus(obj)) {
            return obj;
        }
        else {
            return { code, error };
        }
    }
    catch (err) {
        return { code, error };
    }
}
export default function handleErrors(responses, formatUrl, kind, repl, returnErrors = false) {
    return __awaiter(this, void 0, void 0, function* () {
        const withErrors = yield Promise.all(responses.map((data) => __awaiter(this, void 0, void 0, function* () {
            let errorData = isReturnedError(data)
                ? tryParseAsStatus(data.code, data.error)
                : isStatus(data)
                    ? data
                    : undefined;
            if (errorData) {
                if (isStatus(errorData)) {
                    // if we could not find the requested resource, do some
                    // backup checks, e.g. to see if the namespace and kind
                    // actually exist
                    if (errorData.code === 404 && kind !== 'Namespace') {
                        // double check that the kind and namespace exist
                        const nsUrl = formatUrl(false);
                        const kindUrl = formatUrl(true) + '?limit=1';
                        const opts = { headers: headersForPlainRequest };
                        const [nsData, kindData] = yield Promise.all([
                            fetchFile(repl, nsUrl, opts)
                                .then(_ => _[0])
                                .catch(err => JSON.parse(err.message)),
                            fetchFile(repl, kindUrl, opts)
                                .then(_ => _[0])
                                .catch(err => JSON.parse(err.message))
                        ]);
                        // kubectl seems to report kind non-existence in preference to namespace non-existence
                        if (isStatus(kindData)) {
                            // kind fetch failure; kubectl uses a slightly different error message here. sigh.
                            const error = new Error(`error: the server doesn't have a resource type "${kind}"`);
                            error.code = 404;
                            return error;
                        }
                        else if (isStatus(nsData)) {
                            // namespace fetch failure
                            errorData = nsData;
                        }
                    }
                    const error = new Error(`Error from server (${errorData.reason}): ${errorData.message}`);
                    error.code = errorData.code;
                    return error;
                }
                else {
                    // some other random error, i.e. not a Kubernetes Status error
                    const error = new Error(errorData.error);
                    error.code = errorData.code || 500;
                    return error;
                }
            }
            return data;
        })));
        if (!returnErrors && withErrors.every(isCodedError)) {
            // we didn't get a single good response back
            const error = new Error(withErrors.map(_ => _.message).join('\n'));
            error.code = withErrors[0].code;
            throw error;
        }
        else {
            const init = {
                errors: [],
                ok: [],
                okIndices: [],
                errorIndices: []
            };
            return withErrors.reduce((pair, data, idx) => {
                if (isCodedError(data)) {
                    pair.errors.push(data);
                    pair.errorIndices.push(idx);
                }
                else {
                    pair.ok.push(data);
                    pair.okIndices.push(idx);
                }
                return pair;
            }, init);
        }
    });
}
//# sourceMappingURL=errors.js.map