/*
 * Copyright 2019 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { i18n } from '@kui-shell/core';
import { isSummarizableKubeResource, isKubeResourceWithItsOwnSummary, isDeployment, isEvent, isJob, isNamespace, isNode, isPod, isReplicaSet, isSecret } from '../../../model/resource';
import DeploymentSummary from './impl/Deployment';
import EventSummary from './impl/Event';
import GenericSummary from './impl/Generic';
import JobSummary from './impl/Job';
import NamespaceSummary from './impl/Namespace';
import NodeSummary from './impl/Node';
import PodSummary from './impl/Pod';
import ReplicaSetSummary from './impl/ReplicaSet';
import SecretSummary from './impl/Secret';
const strings = i18n('plugin-kubectl');
/**
 * The content renderer for the summary tab
 *
 */
function renderSummary({ REPL }, resource) {
    return __awaiter(this, void 0, void 0, function* () {
        if (isKubeResourceWithItsOwnSummary(resource)) {
            return resource.summary;
        }
        try {
            const jsyaml = import('js-yaml');
            const map = isPod(resource)
                ? PodSummary(resource)
                : isDeployment(resource)
                    ? DeploymentSummary(resource)
                    : isEvent(resource)
                        ? EventSummary(resource)
                        : isReplicaSet(resource)
                            ? ReplicaSetSummary(resource)
                            : isNamespace(resource)
                                ? NamespaceSummary(resource, REPL)
                                : isNode(resource)
                                    ? NodeSummary(resource)
                                    : isJob(resource)
                                        ? JobSummary(resource)
                                        : isSecret(resource)
                                            ? SecretSummary(resource)
                                            : GenericSummary(resource, REPL);
            // our content is that map, rendered as yaml
            return {
                content: (yield jsyaml).dump(yield map),
                contentType: 'yaml'
            };
        }
        catch (err) {
            if (err.code === 404) {
                return strings('This resource has been deleted');
            }
            else {
                console.error(err);
                return {
                    content: resource.kuiRawData,
                    contentType: 'yaml'
                };
            }
        }
    });
}
/**
 * The Summary mode applies to all KubeResources, and uses
 * `renderContent` to render the view.
 *
 */
const summaryMode = {
    when: isSummarizableKubeResource,
    mode: {
        mode: 'summary',
        label: strings('summary'),
        content: renderSummary,
        // traits:
        defaultMode: true,
        order: -999 // we want this to be placed as the first tab
    }
};
export default summaryMode;
//# sourceMappingURL=index.js.map