/*
 * Copyright 2019 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { Common, CLI, ReplExpect, SidecarExpect, Selectors, Util } from '@kui-shell/test';
import { list, defaultModeForGet, createNS, allocateNS, deleteNS, openSidecarByList, singletonTablesHaveTitle, waitTillNone } from '@kui-shell/plugin-kubectl/tests/lib/k8s/utils';
import * as assert from 'assert';
import { dirname } from 'path';
const ROOT = dirname(require.resolve('@kui-shell/plugin-kubectl/tests/package.json'));
const ns = createNS();
const inNamespace = `-n ${ns}`;
describe(`kubectl deployment ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
    before(Common.before(this));
    after(Common.after(this));
    let res;
    const createIt = () => {
        it('should create deployment from local file', () => __awaiter(this, void 0, void 0, function* () {
            try {
                yield openSidecarByList(this, `kubectl create -f ${ROOT}/data/k8s/deployment.yaml ${inNamespace}`, 'myapp');
            }
            catch (err) {
                return Common.oops(this, true)(err);
            }
        }));
    };
    const listIt = () => {
        it('should list deployments', () => __awaiter(this, void 0, void 0, function* () {
            try {
                const selector = yield list(this, `kubectl get deployment ${inNamespace}`, 'myapp');
                const res = yield Util.openSidecarByClick(this, `${selector} [data-value="myapp"].clickable`, 'myapp', defaultModeForGet);
                const selectorPrefix = selector.replace(Selectors.BY_NAME('myapp'), '');
                if (singletonTablesHaveTitle) {
                    yield this.app.client
                        .$(`${selectorPrefix} .result-table-title`)
                        .then(_ => _.getText())
                        .then(title => assert.ok(title === 'DEPLOYMENT'));
                }
                yield SidecarExpect.yaml({ Name: 'myapp' })(res)
                    .then(() => new Promise(resolve => setTimeout(resolve, 1000)))
                    .then(() => Util.switchToTab('pods')(res))
                    .then(() => this.app.client.$(`${Selectors.SIDECAR_TAB_CONTENT(res.count, res.splitIndex)} table`))
                    .then(_ => _.waitForExist())
                    .then(() => __awaiter(this, void 0, void 0, function* () {
                    if (singletonTablesHaveTitle) {
                        const actualTitle = yield this.app.client
                            .$(`${Selectors.SIDECAR_CUSTOM_CONTENT(res.count, res.splitIndex)} .result-table-title`)
                            .then(_ => _.getText());
                        assert.strictEqual(actualTitle, 'PODS');
                    }
                }));
            }
            catch (err) {
                return Common.oops(this, true)(err);
            }
        }));
    };
    const getPods = () => {
        it('should list pods in deployment, then navigate using Show Owner Reference button', () => __awaiter(this, void 0, void 0, function* () {
            try {
                yield Common.refresh(this);
                const tableRes = yield CLI.command(`kubectl get pod -lapp=drone-app ${inNamespace}`, this.app);
                const selector = yield ReplExpect.okWithCustom({ selector: Selectors.LIST_RESULT_FIRST })(tableRes);
                res = yield Util.openSidecarByClick(this, selector, 'myapp');
                yield Util.clickSidecarModeButton(this, res, 'ownerReference');
                res = ReplExpect.blockAfter(res);
                yield SidecarExpect.open(res)
                    .then(SidecarExpect.kind('ReplicaSet'))
                    .then(SidecarExpect.button({ mode: 'ownerReference', label: 'Show Owner Reference' }));
                yield Util.clickSidecarModeButton(this, res, 'ownerReference');
                res = ReplExpect.blockAfter(res);
                yield SidecarExpect.open(res).then(SidecarExpect.kind('Deployment'));
            }
            catch (err) {
                return Common.oops(this, true)(err);
            }
        }));
    };
    const deleteItByName = () => {
        it('should delete the deployment by name', () => {
            return CLI.command(`kubectl delete deployment myapp ${inNamespace}`, this.app)
                .then(ReplExpect.okWithAny)
                .then(() => waitTillNone('deployment', undefined, 'myapp', undefined, inNamespace))
                .catch(Common.oops(this, true));
        });
    };
    const deleteItByClickingOnButton = () => {
        it('should delete the deployment by clicking on the sidecar delete button', () => __awaiter(this, void 0, void 0, function* () {
            try {
                yield Util.clickSidecarModeButton(this, res, 'delete');
                yield this.app.client.$(Selectors.CONFIRM_DIALOG).then(_ => _.waitForExist());
                yield this.app.client.$(Selectors.CONFIRM_DIALOG_CONFIRM_BUTTON).then(_ => _.click());
                yield waitTillNone('deployment', undefined, 'myapp', undefined, inNamespace);
            }
            catch (err) {
                return Common.oops(this, true)(err);
            }
        }));
    };
    //
    // here start the tests
    //
    allocateNS(this, ns);
    createIt();
    listIt();
    getPods();
    deleteItByName();
    createIt();
    deleteItByClickingOnButton();
    deleteNS(this, ns);
});
//# sourceMappingURL=deployment.js.map