/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
import { Common, CLI, ReplExpect, SidecarExpect, Selectors, Util } from '@kui-shell/test';
import { remotePodYaml, waitForGreen, createNS, allocateNS, deleteNS } from '@kui-shell/plugin-kubectl/tests/lib/k8s/utils';
const commands = ['kubectl'];
if (process.env.NEEDS_OC) {
    commands.push('oc');
}
/** source yaml for creating our pod */
const yaml = remotePodYaml;
/** the yaml file is expected to create a pod with this name */
const podName = 'nginx';
commands.forEach(command => {
    describe(`${command} get pod multi-tab ${process.env.MOCHA_RUN_TARGET || ''}`, function () {
        before(Common.before(this));
        after(Common.after(this));
        Util.closeAllExceptFirstTab.bind(this)();
        /** create pod in the given namespace */
        const createPod = (ns) => {
            it(`should create sample pod from URL via ${command} in namespace ${ns}`, () => {
                return CLI.command(`${command} create -f ${yaml} -n ${ns}`, this.app)
                    .then(ReplExpect.okWithCustom({ selector: Selectors.BY_NAME(podName) }))
                    .then(selector => waitForGreen(this.app, selector))
                    .catch(Common.oops(this, true));
            });
        };
        /** get pod in the given namespace */
        const getPod = (ns, gotRes) => {
            it(`should get pod in namespace ${ns} via ${command}`, () => __awaiter(this, void 0, void 0, function* () {
                try {
                    const res = yield CLI.command(`${command} get pod ${podName} -n ${ns} -o yaml`, this.app)
                        .then(ReplExpect.ok)
                        .then(SidecarExpect.open)
                        .then(SidecarExpect.showing(podName, undefined, undefined, ns));
                    gotRes(res);
                }
                catch (err) {
                    yield Common.oops(this, true)(err);
                }
            }));
        };
        /** create new Tab */
        const newTab = (expectedIdx) => {
            it('new tab via command', () => CLI.command('tab new', this.app)
                .then(() => this.app.client.$(Selectors.TAB_SELECTED_N(expectedIdx)))
                .then(_ => _.waitForDisplayed())
                .then(() => CLI.waitForSession(this)) // should have an active repl
                .catch(Common.oops(this, true)));
        };
        /** switch to the given Tab */
        const switchToTab = (idx) => {
            it(`switch to tab ${idx}`, () => CLI.command(`tab switch ${idx}`, this.app)
                .then(() => this.app.client.$(Selectors.TAB_SELECTED_N(idx)))
                .then(_ => _.waitForDisplayed())
                .catch(Common.oops(this, true)));
        };
        /** expect sidecar to be showing pod in the given namespace */
        const expectShowing = (res, ns) => {
            it(`should be showing pod in namespace ${ns} in the sidecar`, () => {
                return SidecarExpect.showing(podName, undefined, undefined, ns)(res());
            });
        };
        // here is the test script:
        const ns1 = createNS();
        allocateNS(this, ns1);
        const ns2 = createNS();
        allocateNS(this, ns2);
        createPod(ns1);
        createPod(ns2);
        it('should refresh', () => Common.refresh(this));
        Util.closeAllExceptFirstTab.bind(this)();
        let res1;
        let res2;
        getPod(ns1, (res) => (res1 = res));
        newTab(2); // <-- expect the new tab to be the second tab
        getPod(ns2, (res) => (res2 = res));
        switchToTab(1);
        expectShowing(() => res1, ns1);
        switchToTab(2);
        expectShowing(() => res2, ns2);
        deleteNS(this, [ns1, ns2]);
    });
});
//# sourceMappingURL=get-pod-multi-tab.js.map