"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _react = _interopRequireDefault(require("react"));

var _core = require("@kui-shell/core");

var _ = require("../../");

var _KuiContent = _interopRequireDefault(require("./KuiContent"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Notes: asynchronicity in React is a real pain. I realize that
 * `<Suspense />` is supposed to make this easier, but it's still
 * Experimental.
 *
 * What makes this hard: under a given parent component, an `<Eval/>`
 * will only be constructed once, even when the `props` change
 * (i.e. the `constructor` will only be invoked once per parent
 * element). As such, we need to copy the `props.command` to
 * `state.command`, so that we can force a re-eval of the new
 * command. This in turn means we have to manage `state.isLoading`
 * ourselves. Good grief.
 *
 */
class Eval extends _react.default.PureComponent {
  constructor(props) {
    super(props);
    this.state = Eval.getDerivedStateFromProps(props);
    this.startLoading();
  }

  static getDerivedStateFromProps(props, state) {
    if (!state || state.execUUID !== props.execUUID) {
      return {
        execUUID: props.execUUID,
        isLoading: true,
        command: props.command,
        react: undefined,
        spec: undefined,
        content: undefined,
        contentType: props.contentType
      };
    } else {
      return state;
    }
  }

  startLoading() {
    const done = content => {
      this.setState({
        isLoading: false,
        content
      });
    };

    if (typeof this.props.command === 'string') {
      // command string
      this.props.tab.REPL.qexec(this.props.command).then(done);
    } else {
      Promise.resolve(this.props.command(this.props.tab, this.props.response, this.props.args)).then(content => {
        if ((0, _core.isCommandStringContent)(content)) {
          return this.props.tab.REPL.qexec(content.contentFrom);
        } else {
          return content;
        }
      }).then(content => {
        if ((0, _core.isReactProvider)(content)) {
          this.setState({
            isLoading: false,
            react: content
          });
        } else if ((0, _core.isStringWithOptionalContentType)(content)) {
          this.setState({
            isLoading: false,
            content: content.content,
            contentType: content.contentType,
            spec: content.spec
          });
        } else if ((0, _core.isScalarContent)(content)) {
          done(content.content);
        } else {
          done(content);
        }
      });
    }
  }

  render() {
    if (this.state.isLoading) {
      return _react.default.createElement(_.Loading, null);
    }

    const mode = this.state.react ? this.state.react : {
      spec: this.state.spec,
      content: this.state.content,
      contentType: this.state.contentType
    };
    return _react.default.createElement(_KuiContent.default, Object.assign({}, this.props, {
      mode: mode
    }));
  }

}

exports.default = Eval;