"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isFocusable = isFocusable;
exports.default = void 0;

var _react = _interopRequireDefault(require("react"));

var _fs = require("@kui-shell/plugin-bash-like/fs");

var _core = require("@kui-shell/core");

var _Eval = _interopRequireDefault(require("./Eval"));

var _Editor = _interopRequireDefault(require("./Editor"));

var _DiffEditor = _interopRequireDefault(require("./Editor/DiffEditor"));

var _Table = _interopRequireDefault(require("./Table"));

var _Markdown = _interopRequireDefault(require("./Markdown"));

var _HTMLString = _interopRequireDefault(require("./HTMLString"));

var _HTMLDom = _interopRequireDefault(require("./Scalar/HTMLDom"));

var _ = require("../../");

var _RadioTable = _interopRequireDefault(require("../spi/RadioTable"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
class KuiContent extends _react.default.PureComponent {
  constructor(props) {
    super(props);
    this.state = KuiContent.getDerivedStateFromProps(props);
  }

  static getDerivedStateFromProps(props, state) {
    if (!state || state.mode !== props.mode || state.execUUID !== props.execUUID) {
      return Object.assign(state || {}, {
        mode: props.mode,
        execUUID: props.execUUID
      });
    } else {
      return state;
    }
  }

  render() {
    if (!this.props.isActive) {
      return _react.default.createElement(_react.default.Fragment, null);
    }

    const {
      mode
    } = this.state;
    const {
      tab,
      response,
      willUpdateToolbar
    } = this.props;

    if ((0, _core.isStringWithOptionalContentType)(mode)) {
      if (mode.contentType === 'text/html') {
        return _react.default.createElement(_HTMLString.default, {
          content: mode.content
        });
      } else if (mode.contentType === 'text/markdown') {
        if (!mode.content && (0, _fs.isFile)(response)) {
          // then we need to fetch the file content
          return _react.default.createElement(_Eval.default, Object.assign({}, this.props, {
            command: `vfs fslice ${(0, _core.encodeComponent)(response.spec.fullpath)} 0 8192`,
            contentType: mode.contentType
          }));
        } else {
          return _react.default.createElement(_Markdown.default, {
            tab: tab,
            repl: tab.REPL,
            fullpath: (0, _fs.isFile)(response) ? response.spec.fullpath : undefined,
            source: mode.content
          });
        }
      } else {
        return _react.default.createElement(_Editor.default, {
          content: mode,
          readOnly: false,
          willUpdateToolbar: willUpdateToolbar,
          response: response,
          repl: tab.REPL,
          tabUUID: tab.uuid
        });
      }
    } else if ((0, _core.isStringDiffContent)(mode)) {
      return _react.default.createElement(_DiffEditor.default, {
        contentType: mode.contentType,
        originalContent: mode.content.a,
        modifiedContent: mode.content.b,
        response: response,
        renderSideBySide: true,
        tabUUID: tab.uuid
      });
    } else if ((0, _core.isCommandStringContent)(mode)) {
      const key = `${mode.contentFrom} ${mode.contentType}`;
      return _react.default.createElement(_Eval.default, Object.assign({}, this.props, {
        key: key,
        command: mode.contentFrom,
        contentType: mode.contentType
      }));
    } else if ((0, _core.isFunctionContent)(mode)) {
      const command = mode.content;
      return _react.default.createElement(_Eval.default, Object.assign({}, this.props, {
        key: command.toString(),
        command: command
      }));
    } else if ((0, _core.isScalarContent)(mode)) {
      if ((0, _core.isReactProvider)(mode)) {
        return mode.react({
          willUpdateToolbar
        });
      } else if ((0, _core.isRadioTable)(mode.content)) {
        const radioTable = mode.content; // ^^^ Notes: Even though isRadioTable(mode.content) checks the type of mode.content,
        // RadioTableSpi in KuiContext.Consumer doesn't know the type of mode.content is RadioTable and throws error
        // so we have to re-assign mode.content to work around this typescript compile error

        return _react.default.createElement(_.KuiContext.Consumer, null, config => _react.default.createElement(_RadioTable.default, {
          table: radioTable,
          title: !config.disableTableTitle,
          repl: tab.REPL
        }));
      } else if ((0, _core.isTable)(mode.content)) {
        return (0, _Table.default)(tab, tab.REPL, mode.content, false); // ^^^ Notes: typescript doesn't like this, and i don't know why:
        // "is not assignable to type IntrinsicAttributes..."
        // <PaginatedTable {...props} />
      } else if ((0, _core.isHTML)(mode.content)) {
        return _react.default.createElement(_HTMLDom.default, {
          content: mode.content
        });
      } else {
        console.error('Unsupported scalar content', mode);
      }
    }

    return _react.default.createElement("div", {
      className: "oops"
    }, "Unsupported content");
  }

}

exports.default = KuiContent;

function isFocusable(node) {
  return typeof node.doFocus === 'function';
}