"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getStateFromMMR = getStateFromMMR;
exports.default = void 0;

var _debug = _interopRequireDefault(require("debug"));

var _react = _interopRequireDefault(require("react"));

var _reactCore = require("@patternfly/react-core");

var _core = require("@kui-shell/core");

var _Badge = _interopRequireDefault(require("./Badge"));

var _context = _interopRequireDefault(require("../../Client/context"));

var _ToolbarContainer = _interopRequireDefault(require("./ToolbarContainer"));

var _Toolbar = _interopRequireDefault(require("./Toolbar"));

var _BaseSidecarV = _interopRequireDefault(require("./BaseSidecarV2"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/** Lazily load KuiContent; see https://github.com/IBM/kui/issues/3746 */
const KuiContent = _react.default.lazy(() => Promise.resolve().then(() => require('../../Content/KuiContent')));

const debug = (0, _debug.default)('plugin-sidecar/components/TopNavSidecar');

function getStateFromMMR(tab, response) {
  let allModes = response.modes.slice(0); // consult the view registrar for registered view modes
  // relevant to this resource

  const cmd = '';

  if ((0, _core.isResourceWithMetadata)(response)) {
    (0, _core.addRelevantModes)(tab, allModes, cmd, {
      resource: response
    });
  } else if ((0, _core.isResourceByReference)(response)) {
    (0, _core.addRelevantModes)(tab, allModes, cmd, response);
  } // obey the `order` constraints of the modes


  allModes = allModes.sort((a, b) => {
    return (a.order || 0) - (b.order || 0);
  }); // defaultMode: if the response specified one, then look for it;
  // otherwise, use the mode that considers itself default;
  // otherwise use the first

  const tabs = allModes.filter(_ => !(0, _core.isButton)(_));
  const defaultModeFromResponse = response.defaultMode ? tabs.findIndex(_ => _.mode === response.defaultMode) : -1;
  const defaultModeFromModel = defaultModeFromResponse === -1 ? tabs.findIndex(_ => _.defaultMode) : defaultModeFromResponse;
  const defaultMode = defaultModeFromModel === -1 ? 0 : defaultModeFromModel; // re: as any: yay tsc, there are several open issue for this;
  // it's related to isButton using generics

  const buttonsFromRegistrar = allModes.filter(_core.isButton);
  const buttonsFromResponse = response.buttons || [];
  const buttons = buttonsFromResponse.concat(buttonsFromRegistrar); // toolbarText: if the default mode specified one, then use it;
  // otherwise, use the one specified by response

  const toolbarText = tabs[defaultMode] ? tabs[defaultMode].toolbarText : response.toolbarText;
  return {
    currentTabIndex: defaultMode,
    defaultMode,
    tabs,
    toolbarText,
    viewButtons: buttons.filter(_ => !(_.kind === 'drilldown' && _.showRelatedResource)),
    drilldownButtons: buttons.filter(_ => _.kind === 'drilldown' && _.showRelatedResource)
  };
}
/**
 *
 * TopNavSidecar
 * -----------------------
 * | <TitleBar/>         |
 * -----------------------
 * | nameHash?           |
 * | name                |
 * |---------------------|
 * | Tab | Tab |  ...    | <Tab/> from here down
 * |---------------------|
 * | <Toolbar/>          |   <ToolbarContainer/> from here down
 * |---------------------|
 * | <KuiContent/>       |
 * |                     |
 * -----------------------
 *
 */


class TopNavSidecar extends _BaseSidecarV.default {
  constructor(props) {
    super(props);
    this._didUpdateToolbar = this.didUpdateToolbar.bind(this);
    this._onSelect = this.onSelect.bind(this);
    this.state = TopNavSidecar.getDerivedStateFromProps(props);
  }
  /** @return a `HistoryEntry` for the given `Response` */


  static getDerivedStateFromProps(props, state) {
    const {
      tab,
      response
    } = props;

    if (!state || state.response !== response) {
      const args = {
        argsForMode: response.argsForMode,
        argvNoOptions: props.argvNoOptions,
        parsedOptions: props.parsedOptions
      };
      return Object.assign(state || {}, {
        response,
        toolbarText: response.toolbarText,
        args
      }, getStateFromMMR(tab, response));
    } else {
      return state;
    }
  }

  headerBodyStyle() {
    return {
      'flex-direction': 'column'
    };
  }
  /** return the pretty name or unadulterated name from the response */


  prettyName() {
    return this.state.response && (this.state.response.prettyName || (this.state.response.metadata ? this.state.response.metadata.name : undefined));
  }
  /** display the unadulterated name from the response as sidecar header */


  namePart() {
    return this.context.sidecarName === 'heroText' && this.state.response && this.state.response.metadata && this.state.response.metadata.name && _react.default.createElement("div", {
      className: "header-left-bits"
    }, _react.default.createElement("div", {
      className: "sidecar-header-text"
    }, _react.default.createElement("div", {
      className: "sidecar-header-name",
      "data-base-class": "sidecar-header-name"
    }, _react.default.createElement("div", {
      className: "sidecar-header-name-content",
      "data-base-class": "sidecar-header-name-content"
    }, this.state.response.metadata.name))));
  }
  /** ToolbarContainer updated the toolbar */


  didUpdateToolbar(toolbarText) {
    this.setState({
      toolbarText
    });
  }
  /** Special case with no hero name, but badges... we need a filler element */


  fillerNamePart() {
    return _react.default.createElement("div", {
      className: "header-left-bits"
    });
  }
  /** Tell the world that we have changed the focused mode */


  broadcastFocusChange(idx) {
    // de-focus the old mode
    const oldMode = this.current.tabs[this.current.currentTabIndex];

    _core.eventChannelUnsafe.emit(`/mode/focus/off/tab/${this.props.uuid}/mode/${oldMode.mode}`, oldMode); // re-focus the new mode


    const newMode = this.current.tabs[idx];

    _core.eventChannelUnsafe.emit(`/mode/focus/on/tab/${this.props.uuid}/mode/${newMode.mode}`, newMode);
  }
  /** eventKey property for a Tab */


  eventKey(idx) {
    return idx;
  }
  /** idx from the encoded eventKey */


  idxFromEventKey(eventKey) {
    return eventKey;
  }
  /** User has changed selected Tab */


  onSelect(_, eventKey) {
    const idx = this.idxFromEventKey(eventKey); // tell the views that we have changed focus

    this.broadcastFocusChange(idx);
    this.setState(curState => {
      const toolbarText = curState.tabs[idx].toolbarText || curState.toolbarText;
      return Object.assign({}, curState, {
        currentTabIndex: idx,
        toolbarText
      });
    });
  } // first div used to be sidecar-top-stripe


  tabs() {
    return _react.default.createElement("div", {
      className: "kui--sidecar-tabs-container zoomable full-height",
      onClick: this._stopPropagation
    }, _react.default.createElement(_reactCore.Tabs, {
      className: "sidecar-bottom-stripe-mode-bits sidecar-bottom-stripe-button-container kui--sidecar-tabs",
      activeKey: this.eventKey(this.current.currentTabIndex),
      onSelect: this._onSelect
    }, this.current.tabs.map((mode, idx) => _react.default.createElement(_reactCore.Tab, {
      key: mode.mode,
      id: mode.mode,
      eventKey: this.eventKey(idx),
      className: "sidecar-bottom-stripe-button kui--sidecar-tab kui--full-height",
      title: _react.default.createElement(_reactCore.TabTitleText, {
        className: "kui--sidecar-tab-label"
      }, mode.label || mode.mode),
      "data-mode": mode.mode,
      "data-is-selected": idx === this.current.currentTabIndex || undefined,
      onMouseDown: this._preventDefault
    }, this.tabContent(idx)))));
  }

  bodyContent(idx) {
    const mode = this.current.tabs[idx];
    return _react.default.createElement(KuiContent, {
      tab: this.props.tab,
      mode: mode,
      isActive: idx === this.current.currentTabIndex,
      args: this.state.args,
      response: this.state.response,
      execUUID: this.props.execUUID
    });
  }

  tabContent(idx) {
    return _react.default.createElement("div", {
      className: "sidecar-content-container kui--tab-content",
      hidden: idx !== this.current.currentTabIndex || undefined
    }, _react.default.createElement("div", {
      className: "custom-content"
    }, _react.default.createElement(_ToolbarContainer.default, {
      tab: this.props.tab,
      execUUID: this.props.execUUID,
      response: this.state.response,
      args: this.state.args,
      didUpdateToolbar: this._didUpdateToolbar,
      toolbarText: this.state.toolbarText,
      noAlerts: this.current.currentTabIndex !== this.current.defaultMode,
      buttons: this.current.viewButtons
    }, this.bodyContent(idx))));
  }
  /** Return a collection of badge elements */


  badges() {
    const badges = _core.badgeRegistrar.filter(({
      when
    }) => {
      // filter out any irrelevant badges (for this resource)
      try {
        return when(this.state.response);
      } catch (err) {
        debug('warning: registered badge threw an exception during filter', err);
        return false;
      }
    });

    return badges && badges.length > 0 && _react.default.createElement("div", {
      className: "badges"
    }, badges.map(({
      badge
    }, idx) => _react.default.createElement(_Badge.default, {
      key: idx,
      spec: badge,
      tab: this.props.tab,
      response: this.state.response
    })));
  }

  header() {
    const namePart = this.namePart();
    const badges = this.badges();

    if (namePart || badges) {
      return _react.default.createElement("header", {
        className: "sidecar-header"
      }, _react.default.createElement("div", {
        className: "header-main-content"
      }, _react.default.createElement("div", {
        className: "kui--sidecar-header-and-toolbar"
      }, _react.default.createElement("div", {
        className: "header-top-bits"
      }, namePart || this.fillerNamePart(), _react.default.createElement("div", {
        className: "header-right-bits"
      }, _react.default.createElement("div", {
        className: "custom-header-content"
      }, badges))))));
    }
  }

  footer() {
    return _react.default.createElement(_Toolbar.default, {
      bottom: true,
      tab: this.props.tab,
      execUUID: this.props.execUUID,
      response: this.state.response,
      args: this.state.args,
      buttons: this.current.drilldownButtons
    });
  }

  kindBreadcrumb() {
    const {
      kind,
      onclick
    } = this.state.response;
    return {
      label: kind,
      command: onclick && onclick.kind,
      className: 'kui--sidecar-kind'
    };
  }
  /** show name as breadcrumb when not showing context as hero text in sidecar header  */


  nameBreadcrumb() {
    const {
      onclick
    } = this.state.response;
    return {
      label: this.prettyName(),
      command: onclick && onclick.name,
      isCurrentPage: true,
      className: 'kui--sidecar-entity-name'
    };
  }

  versionBreadcrumb() {
    return this.state.response.version ? {
      label: this.state.response.version,
      className: 'kui--version-breadcrumb'
    } : undefined;
  }

  nameHashBreadcrumb() {
    const {
      onclick
    } = this.state.response;
    return {
      label: this.state.response && this.state.response.nameHash,
      command: onclick && onclick.nameHash,
      deemphasize: true,
      className: 'kui--sidecar-entity-name-hash'
    };
  }

  namespaceBreadcrumb() {
    const {
      metadata: {
        namespace
      },
      onclick
    } = this.state.response;
    return {
      label: namespace,
      command: onclick && onclick.namespace,
      deemphasize: true,
      className: 'kui--sidecar-entity-namespace'
    };
  }

  render() {
    if (!this.current || !this.state.response) {
      if (this.props.onRender) {
        this.props.onRender(false);
      }

      return _react.default.createElement("div", null);
    } else if (this.props.onRender) {
      // needs to be async'd; see https://github.com/kubernetes-sigs/kui/issues/7539
      setTimeout(() => this.props.onRender(true));
    }

    const nameBreadCrumbs = this.context.sidecarName === 'breadcrumb' ? [this.nameBreadcrumb(), this.versionBreadcrumb(), this.nameHashBreadcrumb()] : []; // we want to show the titlebar, in order to show the breadcrumbs

    const showingBreadcrumbs = true;

    try {
      const breadcrumbs = [this.kindBreadcrumb()].concat(nameBreadCrumbs).concat(this.namespaceBreadcrumb()).filter(_ => _); // Note: data-view helps with tests

      return _react.default.createElement("div", {
        className: 'kui--sidecar kui--inverted-color-context kui--sidecar-nested ' + this.width(),
        ref: this.dom,
        "data-view": "topnav",
        onClick: this._stopPropagation
      }, showingBreadcrumbs && this.title({
        breadcrumbs
      }), _react.default.createElement("div", {
        className: "kui--sidecar-header-and-body",
        style: {
          flexDirection: 'column'
        }
      }, this.header(), this.tabs(), this.footer()));
    } catch (err) {
      console.error(err);
    }
  }

}

exports.default = TopNavSidecar;
TopNavSidecar.contextType = _context.default;