/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import React from 'react';
import { editor as Monaco, KeyMod, KeyCode } from 'monaco-editor';
import { eventChannelUnsafe, eventBus } from '@kui-shell/core';
import getKuiFontSize from './lib/fonts';
import defaultMonacoOptions from './lib/defaults';
import '../../../../web/scss/components/Editor/Editor.scss';
export default class SimpleEditor extends React.Component {
    constructor(props) {
        super(props);
        // created below in render() via ref={...} -> initMonaco()
        this.state = {
            cleaners: [],
            wrapper: React.createRef(),
            editor: undefined,
            catastrophicError: undefined
        };
    }
    static getDerivedStateFromError(error) {
        return { catastrophicError: error };
    }
    componentDidCatch(error, errorInfo) {
        console.error('catastrophic error in Editor', error, errorInfo);
    }
    shouldComponentUpdate(nextProps, nextState) {
        return !nextState.editor || !!nextState.catastrophicError;
    }
    /** Called whenever we have proposed (props,state); we derive a new State */
    static getDerivedStateFromProps(props, state) {
        if (state.editor) {
            if (state.editor.getValue() !== props.content) {
                state.editor.setValue(props.content);
            }
        }
        return state;
    }
    /** Called when this component is no longer attached to the document */
    componentWillUnmount() {
        this.destroyMonaco();
    }
    /** Called when we no longer need the monaco-editor instance */
    destroyMonaco() {
        this.state.cleaners.forEach(cleaner => cleaner());
    }
    /** When content of the editor changes, call `onContentChange`  */
    static onChange(props, editor) {
        return () => {
            props.onContentChange(editor.getValue());
        };
    }
    /** Register onSave and onCancel keyboard shortcuts */
    registerKeyboardShortcuts(editor) {
        const { onSave, onCancel } = this.props;
        // if the enclosing view has passed in an onSave controller, we
        // will register CtrlCmd+S and Shift+Enter as keyboard shortcuts
        // for this controller
        if (onSave) {
            editor.addCommand(KeyMod.CtrlCmd | KeyCode.KEY_S, () => {
                onSave(editor.getValue());
            });
            editor.addCommand(KeyMod.Shift | KeyCode.Enter, () => {
                onSave(editor.getValue());
            });
        }
        // ibid, but for Escape -> onCancel
        if (onCancel) {
            editor.addCommand(KeyCode.Escape, () => {
                onCancel(editor.getValue());
            });
        }
    }
    /** Called when we have a ready wrapper (monaco's init requires an wrapper */
    initMonaco(props, state) {
        const cleaners = [];
        try {
            // here we instantiate an editor widget
            const providedOptions = {
                value: props.content,
                readOnly: props.readonly !== undefined ? props.readonly : true,
                fontSize: props.fontSize || getKuiFontSize(),
                language: /^(ba)?sh$/.test(props.contentType) ? 'shell' : props.contentType,
                simple: props.simple
            };
            const overrides = { theme: props.light ? 'vs' : 'vs-dark' };
            const options = Object.assign(defaultMonacoOptions(providedOptions), providedOptions, overrides);
            const editor = Monaco.create(state.wrapper.current, options);
            this.registerKeyboardShortcuts(editor);
            if (options.readOnly && props.simple) {
                // if we know 1) the height of the content won't change, and
                // 2) we are running in "simple" mode (this is mostly the case
                // for inline editor components, as opposed to editor
                // components that are intended to fill the full view), then:
                // size the height to fit the content
                const minHeight = this.props.minHeight !== undefined ? this.props.minHeight : 250;
                state.wrapper.current.style.height =
                    Math.min(0.3 * window.innerHeight, Math.max(minHeight, editor.getContentHeight())) + 'px';
            }
            state.wrapper.current['getValueForTests'] = () => {
                return editor.getValue();
            };
            editor.onDidChangeModelContent(SimpleEditor.onChange(props, editor));
            if (!options.readOnly) {
                setTimeout(() => editor.focus());
            }
            const onZoom = () => {
                editor.updateOptions({ fontSize: getKuiFontSize() });
            };
            eventChannelUnsafe.on('/zoom', onZoom);
            cleaners.push(() => eventChannelUnsafe.off('/zoom', onZoom));
            const onTabLayoutChange = () => {
                editor.layout();
            };
            eventBus.onTabLayoutChange(props.tabUUID, onTabLayoutChange);
            cleaners.push(() => eventBus.offTabLayoutChange(props.tabUUID, onTabLayoutChange));
            cleaners.push(() => {
                editor.dispose();
                const model = editor.getModel();
                if (model) {
                    model.dispose();
                }
            });
            this.setState({
                editor,
                cleaners
            });
        }
        catch (err) {
            console.error('Error initing Monaco: ', err);
            this.setState({
                catastrophicError: err
            });
        }
    }
    componentDidMount() {
        if (!this.state.editor && this.state.wrapper.current) {
            // then we are ready to render monaco into the wrapper
            this.initMonaco(this.props, this.state);
        }
        else if (this.props.scrollIntoView && this.state.wrapper.current) {
            this.state.wrapper.current.scrollIntoView();
        }
    }
    render() {
        if (this.state.catastrophicError) {
            return React.createElement("div", { className: "oops" },
                " ",
                this.state.catastrophicError.toString());
        }
        else {
            const className = 'monaco-editor-wrapper' + (this.props.className ? ' ' + this.props.className : '');
            return React.createElement("div", { className: className, ref: this.state.wrapper });
        }
    }
}
//# sourceMappingURL=SimpleEditor.js.map