/*
 * Copyright 2020 The Kubernetes Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { TableStyle, i18n, isTableWithCount, isTableWithTimestamp, isWatchable } from '@kui-shell/core';
import React from 'react';
import { TableComposable } from '@patternfly/react-table';
import sortRow from './sort';
import Card from '../../spi/Card';
import Timeline from './Timeline';
import renderBody from './TableBody';
import renderHeader from './TableHeader';
import SequenceDiagram from './SequenceDiagram';
import Histogram from './Histogram';
import kuiHeaderFromBody from './kuiHeaderFromBody';
import Toolbar from './Toolbar';
import Grid, { findGridableColumn } from './Grid';
/** import the kui theme alignment */
import '../../../../web/scss/components/Table/PatternFly.scss';
import '../../../../web/scss/components/Table/carbon-kui-theme-alignment.scss';
const strings = i18n('plugin-client-common');
export function getBreadcrumbsFromTable(response, prefixBreadcrumbs) {
    const titleBreadcrumb = response.title
        ? [{ label: response.title, className: 'kui--data-table-title kui--sidecar-kind', isCurrentPage: true }]
        : [];
    const _responseCrumbs = typeof response.breadcrumbs === 'function' ? response.breadcrumbs() : response.breadcrumbs;
    const responseCrumbs = !_responseCrumbs
        ? []
        : _responseCrumbs.map(_ => Object.assign({}, _, { className: 'kui--secondary-breadcrumb' }));
    const breadcrumbs = (prefixBreadcrumbs || []).concat(responseCrumbs).concat(titleBreadcrumb);
    return breadcrumbs;
}
/**
 * A DataTable/Pagination pair
 *
 */
export default class PaginatedTable extends React.PureComponent {
    constructor(props) {
        super(props);
        this.defaultPageSize = props.pageSize || 10;
        /** Manage the --kui--table-max-height css variable */
        if (!PaginatedTable.oneTimeInit) {
            PaginatedTable.oneTimeInit = true;
            const adjustMaxHeight = () => {
                const maxHeight = ~~(0.66 * document.body.getBoundingClientRect().height);
                document.body.style.setProperty('--kui--table-max-height', `${maxHeight}px`);
            };
            adjustMaxHeight();
            window.addEventListener('resize', adjustMaxHeight);
        }
        this.state = PaginatedTable.getDerivedStateFromProps(props);
    }
    static getDerivedStateFromProps(props, currentState) {
        try {
            const { header = kuiHeaderFromBody(props.response.body), body, footer, defaultPresentation } = props.response;
            props.response.header = header;
            if (props.config && props.config.drilldownTo) {
                props.response.drilldownTo = props.config.drilldownTo;
            }
            const asGrid = ((!defaultPresentation || defaultPresentation === 'grid') &&
                props.asGrid &&
                findGridableColumn(props.response) >= 0) ||
                defaultPresentation === 'grid';
            const asSequence = (!asGrid && PaginatedTable.hasSequenceButton(props) && props.response.body.length > 1) ||
                defaultPresentation === 'sequence-diagram';
            const defaults = {
                page: 1,
                pageSize: 10,
                asTimeline: false
            };
            const newState = {
                body,
                asGrid,
                footer,
                header,
                asSequence,
                asHistogram: defaultPresentation === 'histogram',
                activeSortIdx: -1,
                activeSortDir: undefined,
                response: props.response,
                pageSize: props.pageSize || defaults.pageSize
            };
            if (!currentState) {
                // new instance of this component
                return Object.assign(defaults, newState);
            }
            else if (newState.response !== currentState.response) {
                // user made changes to the existing props instance
                return Object.assign(defaults, currentState, newState);
            }
            else {
                // different props
                return Object.assign(defaults, newState, currentState);
            }
        }
        catch (err) {
            console.error('Internal error preparing PaginatedTable', err);
        }
    }
    topToolbar(lightweightTables = false) {
        // 1) If we started as a table, and are now a grid, then show
        // "Status Grid", otherwise:
        // 2) only for client w/o disableTableTitle, show a breadcrumb
        const breadcrumbs = !lightweightTables && !this.props.asGrid && this.state.asGrid
            ? [{ label: strings('Status Grid') }]
            : !this.props.asGrid && !this.props.title
                ? []
                : getBreadcrumbsFromTable(this.props.response, this.props.prefixBreadcrumbs);
        // ... except when we're displaying asGrid by default
        if (!(!lightweightTables && !this.props.asGrid && this.state.asGrid)) {
            const nRows = this.props.response.body.length;
            if (!lightweightTables && breadcrumbs.length > 0 && (nRows > 1 || isWatchable(this.props.response))) {
                breadcrumbs.push({
                    label: nRows === 1 ? strings('nRows1') : strings('nRows', nRows),
                    className: 'kui--secondary-breadcrumb kui--nrows-breadcrumb'
                });
            }
        }
        if (breadcrumbs.length > 0) {
            return (React.createElement(Toolbar, { className: "kui--data-table-toolbar-top", breadcrumbs: breadcrumbs.length > 0 && breadcrumbs, repl: this.props.repl }));
        }
    }
    isPaginated() {
        return false;
        /**
         * return (
         * this.props.paginate !== undefined &&
         * this.props.paginate !== false &&
         * !this.state.asGrid &&
         * (this.props.paginate === true || this.state.rows.length > this.props.paginate)
         * )
         */
    }
    hasFooterLines() {
        return this.state.footer && this.state.footer.length > 0;
    }
    footerLines() {
        const nRows = this.props.isPartOfMiniSplit ? -1 : -6;
        return this.state.footer ? this.state.footer.slice(nRows) : undefined;
    }
    static hasSequenceButton(props) {
        return isTableWithTimestamp(props.response);
    }
    static hasHistogramButton(props) {
        return isTableWithCount(props.response);
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    bottomToolbar(lightweightTables = false) {
        const gridableColumn = findGridableColumn(this.props.response);
        const hasSequenceButton = PaginatedTable.hasSequenceButton(this.props);
        const hasHistogramButton = PaginatedTable.hasHistogramButton(this.props);
        const hasTimelineButton = false; // disabled for now, see https://github.com/IBM/kui/issues/5864
        const needsBottomToolbar = this.caption() ||
            this.isPaginated() ||
            (gridableColumn >= 0 && (this.props.response.body.length > 1 || isWatchable(this.props.response))) ||
            isTableWithTimestamp(this.props.response) ||
            isTableWithCount(this.props.response);
        return (React.createElement(React.Fragment, null,
            this.hasFooterLines() && React.createElement(Toolbar, { stream: this.footerLines(), repl: this.props.repl }),
            this.props.toolbars && needsBottomToolbar && (React.createElement(Toolbar, { className: "kui--data-table-toolbar-bottom", repl: this.props.repl, asGrid: this.state.asGrid, gridableColumn: gridableColumn, setAsGrid: (asGrid) => {
                    this.setState({ asGrid });
                    if (asGrid) {
                        this.props.response.defaultPresentation = 'grid';
                    }
                }, paginate: this.isPaginated(), setPage: (page) => this.setState({ page }), page: this.state.page, totalItems: this.state.body.length, pageSize: this.state.pageSize, hasSequenceButton: hasSequenceButton, asSequence: this.state.asSequence, setAsSequence: (asSequence) => {
                    this.setState({ asSequence });
                    if (asSequence) {
                        this.props.response.defaultPresentation = 'sequence-diagram';
                    }
                }, hasHistogramButton: hasHistogramButton, asHistogram: this.state.asHistogram, setAsHistogram: (asHistogram) => {
                    this.setState({ asHistogram });
                    if (asHistogram) {
                        this.props.response.defaultPresentation = 'histogram';
                    }
                }, hasTimelineButton: hasTimelineButton, asTimeline: this.state.asTimeline, caption: this.caption() || undefined, setAsTimeline: (asTimeline) => {
                    this.setState({ asTimeline });
                    if (asTimeline) {
                        this.props.response.defaultPresentation = 'timeline';
                    }
                } }))));
    }
    /** E.g. last updated time for live tables */
    // eslint-disable-next-line @typescript-eslint/no-empty-function
    caption() { }
    grid(visibleRows) {
        return (React.createElement(Grid, { tab: this.props.tab, repl: this.props.repl, response: this.props.response, visibleRows: visibleRows, justUpdated: this.justUpdatedMap() }));
    }
    justUpdatedMap() {
        return this.state.body.reduce(M => {
            // FIXME if (_.justUpdated) M[_.rowKey] = true
            return M;
        }, {});
    }
    table() {
        const { tab, repl } = this.props;
        const { page } = this.state;
        const response = this.props.response;
        const { body, header } = this.state;
        const variant = 'compact';
        /* this.props.response.style === TableStyle.Heavy
          ? null
          : this.props.response.style === undefined || this.props.response.style === TableStyle.Medium
          ? null
          : 'compact' */
        const isSortable = response.body.length > 1;
        const onSort = (key, cidx, clickColIdx, clickDir) => {
            response.body.sort((a, b) => sortRow(a, b, key, cidx, clickDir));
            this.setState({
                body,
                activeSortDir: clickDir,
                activeSortIdx: clickColIdx
            });
        };
        const dataTable = (visibleRows, offset = 0) => (React.createElement("div", { className: 'kui--data-table-container' +
                (this.props.title ? ' kui--data-table-container-with-toolbars' : '') +
                (this.props.response.title || this.props.response.breadcrumbs
                    ? ' kui--data-table-container-with-title'
                    : '') +
                (isSortable ? ' kui--data-table-sortable' : ''), "data-is-empty": response.body.length === 0 },
            React.createElement(TableComposable, { className: "kui--table-like-wrapper", variant: variant, isStickyHeader: true, gridBreakPoint: "" },
                header &&
                    renderHeader(header, isSortable, this.state.activeSortIdx, this.state.activeSortDir, onSort.bind(this)),
                renderBody(response, this.justUpdatedMap(), tab, repl, offset))));
        const paginated = this.isPaginated();
        return dataTable(!paginated ? body : body.slice((page - 1) * this.state.pageSize, page * this.state.pageSize), !paginated ? 0 : (page - 1) * this.state.pageSize);
    }
    sequence() {
        return React.createElement(SequenceDiagram, Object.assign({}, this.props, { isWatching: this.isWatching() }));
    }
    histogram() {
        return React.createElement(Histogram, Object.assign({}, this.props, { isWatching: this.isWatching() }));
    }
    timeline() {
        return React.createElement(Timeline, Object.assign({}, this.props));
    }
    /** @return whether the table is currently "live", and responding to updates from the controller */
    isWatching() {
        return false;
    }
    content(includeToolbars = false, lightweightTables = false) {
        if (this.props.onRender) {
            this.props.onRender(true);
        }
        return (React.createElement(React.Fragment, null,
            includeToolbars && this.topToolbar(lightweightTables),
            this.state.asGrid
                ? this.grid(this.state.body)
                : this.state.asSequence
                    ? this.sequence()
                    : this.state.asHistogram
                        ? this.histogram()
                        : this.state.asTimeline
                            ? this.timeline()
                            : this.table(),
            includeToolbars && this.bottomToolbar(lightweightTables)));
    }
    dataAttrs() {
        return {};
    }
    render() {
        if (!this.state) {
            return React.createElement("div", { className: "oops" }, "Internal Error");
        }
        else {
            const lightweightTables = this.props.response.style === TableStyle.Light;
            const className = 'kui--data-table-wrapper' +
                (this.state.asGrid ? ' kui--data-table-as-grid' : '') +
                (this.state.asSequence ? ' kui--data-table-as-sequence' : '') +
                (this.state.asTimeline ? ' kui--data-table-as-timeline' : '') +
                (lightweightTables ? ' kui--data-table-wrapper-lightweight flex-fill' : '');
            return (React.createElement("div", Object.assign({ className: className }, this.dataAttrs()),
                React.createElement("div", { className: "kui--screenshotable" }, lightweightTables ? (this.content(true, true)) : (React.createElement(Card, { header: this.topToolbar(), footer: this.bottomToolbar(), className: "kui--table-card" }, this.content())))));
        }
    }
}
/**
 * A lock that governs the --kui--table-max-height
 * initialization. See https://github.com/IBM/kui/issues/5206
 *
 */
PaginatedTable.oneTimeInit = false;
//# sourceMappingURL=PaginatedTable.js.map