import { CommandStartEvent, CommandCompleteEvent, KResponse, ScalarResponse } from '@kui-shell/core';
export declare const enum BlockState {
    Active = "repl-active",
    Cancelled = "cancelled",
    Empty = "empty",
    Error = "error",
    Processing = "processing",
    ValidResponse = "valid-response"
}
/** Traits that Blocks might have */
declare type WithCWD = {
    cwd: string;
};
declare type WithUUID = {
    execUUID: string;
};
declare type WithOriginalExecUUID = {
    originalExecUUID: string;
};
declare type WithCommand = {
    command: string;
    isExperimental?: boolean;
} & WithCWD;
declare type WithStartTime = {
    startTime: number;
};
declare type WithState<S extends BlockState> = {
    state: S;
};
declare type WithResponse<R extends KResponse> = {
    response: R;
} & WithStartTime;
declare type WithValue = {
    value: string;
};
declare type WithAnnouncement = {
    isAnnouncement: boolean;
};
declare type WithPreferences = {
    outputOnly?: boolean;
};
declare type WithCommandStart = {
    startEvent: CommandStartEvent;
};
declare type WithCommandComplete = {
    completeEvent: CommandCompleteEvent;
};
declare type WithRerun = {
    isRerun: true;
    newExecUUID: string;
} & WithOriginalExecUUID;
declare type WithReplay = {
    isReplay: boolean;
};
/** The canonical types of Blocks, which mix up the Traits as needed */
declare type ActiveBlock = WithState<BlockState.Active> & WithCWD & Partial<WithValue>;
export declare type AnnouncementBlock = WithState<BlockState.ValidResponse> & WithResponse<ScalarResponse> & WithCWD & WithAnnouncement;
declare type EmptyBlock = WithState<BlockState.Empty> & WithCWD & Partial<WithCommand> & Partial<WithCommandComplete>;
declare type ErrorBlock = WithState<BlockState.Error> & WithCommand & WithResponse<Error> & WithUUID & WithHistoryIndex & WithCommandStart & Partial<WithRerun> & WithReplay & WithCommandComplete;
declare type OkBlock = WithState<BlockState.ValidResponse> & WithCommand & WithResponse<KResponse> & WithUUID & WithHistoryIndex & WithCommandStart & WithCommandComplete & Partial<WithRerun> & WithReplay & WithPreferences;
export declare type ProcessingBlock = WithState<BlockState.Processing> & WithCommand & WithUUID & WithStartTime & Partial<WithOriginalExecUUID> & WithReplay & WithCommandStart;
declare type CancelledBlock = WithState<BlockState.Cancelled> & WithCWD & WithCommand & WithUUID & WithStartTime;
export declare type CompleteBlock = OkBlock | ErrorBlock;
/** Blocks with an association to the History model */
declare type WithHistoryIndex = {
    historyIdx: number;
};
/** FinishedBlocks are either ok, error, or cancelled */
export declare type FinishedBlock = OkBlock | ErrorBlock | CancelledBlock | EmptyBlock;
export declare type BlockModel = ProcessingBlock | FinishedBlock | CancelledBlock | ActiveBlock | AnnouncementBlock;
export default BlockModel;
declare type CustomError = {
    name: string;
    message: string;
};
declare type ErrorLike = Error | CustomError;
export declare function isError(response: KResponse): response is ErrorLike;
export declare function isProcessing(block: BlockModel): block is ProcessingBlock;
export declare function isActive(block: BlockModel): block is ActiveBlock;
/** @return true if the given `block` is `Active` and differs from the other `oblock` */
export declare function isActiveAndDifferent(block: BlockModel, oblock: BlockModel): block is ActiveBlock;
export declare function isCancelled(block: BlockModel): block is CancelledBlock;
export declare function isEmpty(block: BlockModel): block is EmptyBlock;
export declare function isOk(block: BlockModel): block is OkBlock;
export declare function isOops(block: BlockModel): block is ErrorBlock;
export declare function isFinished(block: BlockModel): block is FinishedBlock;
export declare function hasCommand(block: BlockModel & Partial<WithCommand>): block is BlockModel & Required<WithCommand>;
export declare function isAnnouncement(block: BlockModel): block is AnnouncementBlock;
export declare function hasUUID(block: BlockModel & Partial<WithUUID>): block is BlockModel & Required<WithUUID>;
export declare function hasValue(block: BlockModel): block is BlockModel & Required<WithValue>;
/** Transform to Active */
export declare function Active(initialValue?: string): ActiveBlock;
/** Transform to AnnouncementBlock */
export declare function Announcement(response: ScalarResponse): AnnouncementBlock;
export declare function isRerunable(block: BlockModel): block is RerunableBlock;
export declare function isBeingRerun(block: BlockModel): block is BlockBeingRerun;
export declare function hasOriginalUUID(block: BlockModel | WithOriginalExecUUID): block is WithOriginalExecUUID;
export declare function hasBeenRerun(block: BlockModel): boolean;
/** Transform to Processing */
export declare function Processing(block: BlockModel, startEvent: CommandStartEvent, isExperimental?: boolean, isReplay?: boolean): ProcessingBlock;
/** Transform to Empty */
export declare function Empty(block: BlockModel, typedSoFar?: string, completeEvent?: CommandCompleteEvent): EmptyBlock;
/** Transform to Cancelled */
export declare function Cancelled(block: BlockModel, typedSoFar?: string, completeEvent?: CommandCompleteEvent): CancelledBlock | EmptyBlock;
/** Transform to Finished */
export declare function Finished(block: ProcessingBlock | BlockBeingRerun, event: CommandCompleteEvent, outputOnly?: boolean, isReplay?: boolean): FinishedBlock;
export declare function isOutputOnly(block: BlockModel): boolean;
/** @return whether the block as a startEvent trait */
export declare function hasStartEvent(block: BlockModel): block is BlockModel & WithCommandStart;
/** @return whether the block has a completeEvent trait */
export declare function isWithCompleteEvent(block: BlockModel): block is CompleteBlock;
/** @return whether the block has pipeStages information; older snapshots may not */
export declare function hasPipeStages(block: BlockModel): boolean;
/** @return whether the block is from replay */
export declare function isReplay(block: BlockModel): boolean;
/** A Block may be Rerunable. If so, then it can be transitioned to the BlockBeingRerun state. */
declare type RerunableBlock = CompleteBlock;
declare type BlockBeingRerun = RerunableBlock & WithRerun;
/** Transform a RerunableBlock to one in the BlockBeingRerun state */
export declare function Rerun(block: RerunableBlock, newStartEvent?: CommandStartEvent, newCommand?: string, newStartTime?: number): RerunableBlock & Required<WithRerun>;
